% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/write.sav.R
\name{write.sav}
\alias{write.sav}
\title{Write SPSS File}
\usage{
write.sav(x, file = "SPSS_Data.sav", var.attr = NULL, pspp.path = NULL,
          digits = 2, write.csv = FALSE, sep = c(";", ","), na = "",
          write.sps = FALSE, check = TRUE)
}
\arguments{
\item{x}{a matrix or data frame to be written in SPSS, vectors are
coerced to a data frame.}

\item{file}{a character string naming a file with or without file extension
'.sav', e.g., \code{"SPSS_Data.sav"} or \code{"SPSS_Data"}.}

\item{var.attr}{a matrix or data frame with variable attributes used in the
SPSS file, only 'variable labels' (column name \code{label}),
'value labels' column name \code{values}, and 'user-missing values'
column name \code{missing} are supported (see 'Details').}

\item{pspp.path}{a character string indicating the path where the PSPP folder
is located on the computer, e.g.\code{C:/Program Files/PSPP/}.}

\item{digits}{an integer value indicating the number of decimal places shown
in the SPSS file for non-integer variables.}

\item{write.csv}{logical: if \code{TRUE}, CSV file is written along with the
SPSS file.}

\item{sep}{a character string for specifying the CSV file, either \code{";"}
for the separator and \code{"."}
for the decimal point (default, i.e. equivalent to \code{write.csv2})
or \code{"."} for the decimal point and \code{","} for the
separator (i.e. equivalent to \code{write.csv}), must be one
of both \code{";"} (default) or \code{","}.}

\item{na}{a character string for specifying missing values in the CSV file.}

\item{write.sps}{logical: if \code{TRUE}, SPSS syntax is written along with
the SPSS file when using PSPP.}

\item{check}{logical: if \code{TRUE}, variable attributes specified in the
argument \code{var.attr} is checked.}
}
\description{
This function writes a data frame or matrix into a SPSS file by either using the
\code{write_sav()} function in the \pkg{haven} package by Hadley Wickham and Evan
Miller (2019) or the free software \emph{PSPP}.
}
\details{
If arguments \code{pspp.path} is not specified (i.e., \code{pspp.path = NULL}),
\code{write_sav()} function in the \pkg{haven} is used. Otherwise the object \code{x}
is written as CSV file, which is subsequently imported into SPSS using the free
software \emph{PSPP} by executing a SPSS syntax written in R. Note that \emph{PSPP}
needs to be installed on your computer when using the \code{pspp.path} argument.

A SPSS file with 'variable labels', 'value labels', and 'user-missing values' is
written by specifying the \code{var.attr} argument. Note that the number of rows
in the matrix or data frame specified in \code{var.attr} needs to match with the
number of columns in the data frame or matrix specified in \code{x}, i.e., each
row in \code{var.attr} represents the variable attributes of the corresponding
variable in \code{x}. In addition, column names of the matrix or data frame
specified in \code{var.attr} needs to be labeled as \code{label} for 'variable
labels, \code{values} for 'value labels', and \code{missing} for 'user-missing
values'.

Labels for the values are defined in the column \code{values} of the matrix or
data frame in \code{var.attr} using the equal-sign (e.g., \code{0 = female}) and
are separated by a semicolon (e.g., \code{0 = female; 1 = male}).

User-missing values are defined in the column \code{missing} of the matrix or
data frame in \code{var.attr}, either specifying one user-missing value (e.g.,
\code{-99}) or more than one but up to three user-missing values separated
by a semicolon (e.g., \code{-77; -99}.
}
\note{
Part of the function using \emph{PSPP} was adapted from the \code{write.pspp()}
function in the \pkg{miceadds} package by Alexander Robitzsch, Simon Grund and
Thorsten Henke (2019).
}
\examples{
\dontrun{
dat <- data.frame(id = 1:5,
                  gender = c(NA, 0, 1, 1, 0),
                  age = c(16, 19, 17, NA, 16),
                  status = c(1, 2, 3, 1, 4),
                  score = c(511, 506, 497, 502, 491))

# Example 1: Write SPSS file using the haven package
write.sav(dat, file = "Dataframe_haven.sav")

# Example 2: Write SPSS file using PSPP,
# write CSV file and SPSS syntax along with the SPSS file
write.sav(dat, file = "Dataframe_PSPP.sav", pspp.path = "C:/Program Files/PSPP",
          write.csv = TRUE, write.sps = TRUE)

# Example 3: Specify variable attributes
# Note that it is recommended to manually specify the variables attritbues in a CSV or
# Excel file which is subsequently read into R
attr <- data.frame(# Variable names
                   var = c("id", "gender", "age", "status", "score"),
                   # Variable labels
                   label = c("Identification number", "Gender", "Age in years",
                             "Migration background", "Achievement test score"),
                   # Value labels
                   values = c("", "0 = female; 1 = male", "",
                              "1 = Austria; 2 = former Yugoslavia; 3 = Turkey; 4 = other",
                              ""),
                   # User-missing values
                   missing = c("", "-99", "-99", "-99", "-99"), stringsAsFactors = FALSE)

# Example 4: Write SPSS file with variable attributes using the haven package
write.sav(dat, file = "Dataframe_haven_Attr.sav", var.attr = attr)

# Example 5: Write SPSS with variable attributes using PSPP
write.sav(dat, file = "Dataframe_PSPP_Attr.sav", var.attr = attr,
          pspp.path = "C:/Program Files/PSPP")
}
}
\references{
GNU Project (2018). \emph{GNU PSPP for GNU/Linux} (Version 1.2.0).
Boston, MA: Free Software Foundation. https://www.gnu.org/software/pspp/

Wickham H., & Miller, E. (2019). \emph{haven: Import and Export 'SPSS', 'Stata'
and 'SAS' Files}. R package version 2.2.0.

Robitzsch, A., Grund, S., & Henke, T. (2019). \emph{miceadds: Some additional
multiple imputation functions, especially for mice}. R package version 3.4-17.
}
\seealso{
\code{\link{read.sav}}, \code{\link{write.xlsx}}, \code{\link{write.dta}},
\code{\link{write.mplus}}
}
\author{
Takuya Yanagida \email{takuya.yanagida@univie.ac.at}
}
