% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/multilevel.r2.manual.R
\name{multilevel.r2.manual}
\alias{multilevel.r2.manual}
\title{R-Squared Measures for Multilevel and Linear Mixed Effects Models by Rights
and Sterba (2019), Manually Inputting Parameter Estimates}
\usage{
multilevel.r2.manual(data, within = NULL, between = NULL, random = NULL,
                     gamma.w = NULL, gamma.b = NULL, tau, sigma2,
                     intercept = TRUE, center = TRUE, digits = 3,
                     plot = FALSE, gray = FALSE, start = 0.15, end = 0.85,
                     color = c("#D55E00", "#0072B2", "#CC79A7", "#009E73", "#E69F00"),
                     check = TRUE, output = TRUE)
}
\arguments{
\item{data}{a matrix or data frame with the level-1 and level-2 predictors
and outcome variable used in the model.}

\item{within}{a character vector with the variable names in \code{data} or
numeric vector with numbers corresponding to the columns in
\code{data} of the level-1 predictors used in the model. If
none used, set to \code{NULL}.}

\item{between}{a character vector with the variable names in \code{data} or
numeric vector with numbers corresponding to the columns in
\code{data} of the level-2 predictors used in the model. If
none used, set to \code{NULL}.}

\item{random}{a character vector with the variable names in \code{data} or
numeric vector with numbers corresponding to the columns in
\code{data} of the level-1 predictors that have random slopes
in the model. If no random slopes specified, set to \code{NULL}.}

\item{gamma.w}{a numeric vector of fixed slope estimates for all level-1
predictors, to be entered in the order of the predictors
listed in the argument \code{within}.}

\item{gamma.b}{a numeric vector of the intercept and fixed slope estimates
for all level-2predictors, to be entered in the order of the
predictors listed in the argument \code{between}. Note that
the first element is the parameter estimate for the intercept
if \code{intercept = TRUE}.}

\item{tau}{a matrix indicating the random effects covariance matrix, the
first row/column denotes the intercept variance and covariances
(if intercept is fixed, set all to 0) and each subsequent
row/column denotes a given random slope's variance and covariances
(to be entered in the order listed in the argument \code{random}).}

\item{sigma2}{a numeric value indicating the level-1 residual variance.}

\item{intercept}{logical: if \code{TRUE} (default), the first element in the
\code{gamma.b} is assumed to be the fixed intercept estimate;
if set to \code{FALSE}, the first element in the argument
\code{gamma.b} is assumed to be the first fixed level-2
predictor slope.}

\item{center}{logical: if \code{TRUE} (default), all level-1 predictors are
assumed to be cluster-mean-centered and the function will
output all decompositions; if set to \code{FALSE}, function
will output only the total decomposition.}

\item{digits}{an integer value indicating the number of decimal places to be
used.}

\item{plot}{logical: if \code{TRUE}, bar chart showing the decomposition
of scaled total, within-cluster, and between-cluster outcome
variance into five (total), three (within-cluster), and two
(between-cluster) proportions is drawn. Note that the \pkg{ggplot2}
package is required to draw the bar chart.}

\item{gray}{logical: if \code{TRUE}, graphical parameter to draw the bar
chart in gray scale.}

\item{start}{a numeric value between 0 and 1, graphical parameter to specify
the gray value at the low end of the palette.}

\item{end}{a numeric value between 0 and 1, graphical parameter to specify
the gray value at the high end of the palette.}

\item{color}{a character vector, graphical parameter indicating the color
of bars in the bar chart in the following order: Fixed slopes
(Within), Fixed slopes (Between), Slope variation (Within),
Intercept variation (Between), and Residual (Within). By default,
colors from the colorblind-friendly palettes are used.}

\item{check}{logical: if \code{TRUE}, argument specification is checked.}

\item{output}{logical: if \code{TRUE}, output is shown on the console.}
}
\value{
Returns an object of class \code{misty.object}, which is a list with following
entries:
  \item{\code{call}}{function call}
  \item{\code{type}}{type of analysis}
  \item{\code{data}}{matrix or data frame specified in \code{data}}
  \item{\code{plot}}{ggplot2 object for plotting the results}
  \item{\code{args}}{specification of function arguments}
  \item{\code{result}}{list with result tables, i.e., \code{decomp} for the
                       decomposition, \code{total} for total R2 measures,
                       \code{within} for the within-cluster R2 measures, and
                       \code{between}} for the between-cluster R2 measures.
}
\description{
This function computes R-squared measures by Rights and Sterba (2019) for
multilevel and linear mixed effects models by manually inputting parameter
estimates.
}
\details{
A number of R-squared measures for multilevel and linear mixed effects models
have been developed in the methodological literature (see Rights & Sterba, 2018).
R-squared measures by Rights and Sterba (2019) provide an integrative framework
of R-squared measures for multilevel and linear mixed effects models with random
intercepts and/or slopes. Their measures are based on partitioning model implied
variance from a single fitted model, but they provide a full partitioning of
the total outcome variance to one of five specific sources. See the help page
of the \code{\link{multilevel.r2}} function for more details.
}
\note{
This function is calls the \code{r2mlm_manual()} function in the \pkg{r2mlm}
package by Mairead Shaw, Jason Rights, Sonya Sterba, and Jessica Flake.
}
\examples{
\dontrun{
# Load misty, lme4, nlme, and ggplot2 package
library(misty)
library(lme4)

# Load data set "Demo.twolevel" in the lavaan package
data("Demo.twolevel", package = "lavaan")

#---------------------------

# Cluster mean centering, center() from the misty package
Demo.twolevel$x2.c <- center(Demo.twolevel$x2, type = "CWC",
                             cluster = Demo.twolevel$cluster)

# Compute group means, cluster.scores() from the misty package
Demo.twolevel$x2.b <- cluster.scores(Demo.twolevel$x2,
                                     cluster = Demo.twolevel$cluster)

# Estimate random intercept model using the lme4 package
mod1 <- lmer(y1 ~ x2.c + x2.b + w1 + (1| cluster), data = Demo.twolevel,
             REML = FALSE, control = lmerControl(optimizer = "bobyqa"))

# Estimate random intercept and slope model using the lme4 package
mod2 <- lmer(y1 ~ x2.c + x2.b + w1 + (1 + x2.c | cluster), data = Demo.twolevel,
             REML = FALSE, control = lmerControl(optimizer = "bobyqa"))

#---------------------------
# Random intercept model

# Fixed slope estimates
fixef(mod1)

# Random effects variance-covariance matrix
as.data.frame(VarCorr(mod1))

# R-squared measures according to Rights and Sterba (2019)
multilevel.r2.manual(data = Demo.twolevel,
                     within = "x2.c", between = c("x2.b", "w1"),
                     gamma.w = 0.41127956,
                     gamma.b = c(0.01123245, -0.08269374, 0.17688507),
                     tau = 0.9297401,
                     sigma2 = 1.813245794)

#---------------------------
# Random intercept and slope model

# Fixed slope estimates
fixef(mod2)

# Random effects variance-covariance matrix
as.data.frame(VarCorr(mod2))

# R-squared measures according to Rights and Sterba (2019)
multilevel.r2.manual(data = Demo.twolevel,
                     within = "x2.c", between = c("x2.b", "w1"), random = "x2.c",
                     gamma.w = 0.41127956,
                     gamma.b = c(0.01123245, -0.08269374, 0.17688507),
                     tau = matrix(c(0.931008649, 0.004110479, 0.004110479, 0.017068857), ncol = 2),
                     sigma2 = 1.813245794)
}
}
\references{
Rights, J. D., & Cole, D. A. (2018). Effect size measures for multilevel models
in clinical child and adolescent research: New r-squared methods and recommendations.
\emph{Journal of Clinical Child and Adolescent Psychology, 47}, 863-873.
 https://doi.org/10.1080/15374416.2018.1528550

Rights, J. D., & Sterba, S. K. (2019). Quantifying explained variance in multilevel
models: An integrative framework for defining R-squared measures. \emph{Psychological Methods, 24},
309-338. https://doi.org/10.1037/met0000184
}
\seealso{
\code{\link{multilevel.r2}}, \code{\link{multilevel.cor}},
\code{\link{multilevel.descript}}, \code{\link{multilevel.icc}},
\code{\link{multilevel.indirect}}
}
\author{
Jason D. Rights, Sonya K. Sterba, Jessica K. Flake, and Takuya Yanagida
}
