% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/submit_lpjml.R
\name{submit_lpjml}
\alias{submit_lpjml}
\title{Submit LPJmL model simulation to SLURM}
\usage{
submit_lpjml(
  x,
  model_path,
  sim_path = NULL,
  group = "lpjml",
  sclass = "short",
  ntasks = 256,
  wtime = "",
  blocking = "",
  no_submit = FALSE,
  output_path = NULL
)
}
\arguments{
\item{x}{A \link[tibble]{tibble} with at least one column named \code{"sim_name"}.
Each simulation gets a separate row. An optional run parameter
\code{"dependency"} is used for subsequent simulations (see details).
\code{\link[=write_config]{write_config()}} returns a tibble in the required
format. OR provide a character string (vector) with the file name of one or
multiple generated config file(s).}

\item{model_path}{Character string providing the path to LPJmL
(equal to \code{LPJROOT} environment variable).}

\item{sim_path}{Character string defining path where all simulation data are
written, including output, restart and configuration files. If \code{NULL},
\code{model_path} is used. See also \link{write_config}}

\item{group}{Character string defining the user group for which the job is
submitted. Defaults to \code{"lpjml"}.}

\item{sclass}{Character string defining the job classification. Available
options at PIK: \code{c("short", "medium", "long", "priority", "standby", "io")}
More information at \url{https://www.pik-potsdam.de/en}. Defaults to \code{"short"}.}

\item{ntasks}{Integer defining the number of tasks/threads. More information
at \url{https://www.pik-potsdam.de/en} and \url{https://slurm.schedmd.com}.
Defaults to \code{256}.}

\item{wtime}{Character string defining the time limit. Setting a lower time
limit than the maximum runtime for \code{sclass} can reduce the wait time in the
SLURM job queue. More information at \url{https://www.pik-potsdam.de/en} and
\url{https://slurm.schedmd.com}.}

\item{blocking}{Integer defining the number of cores to be blocked. More
information at \url{https://www.pik-potsdam.de/en} and
\url{https://slurm.schedmd.com}.}

\item{no_submit}{Logical. Set to \code{TRUE} to test if \code{x} set correctly or
\code{FALSE} to actually submit job to SLURM.}

\item{output_path}{Argument is deprecated as of version 1.0; use sim_path
instead.}
}
\value{
See \code{x}, extended by columns \code{"type"}, \code{"job_id"} and \code{"status"}.
}
\description{
LPJmL simulations are submitted to SLURM using \code{"config*.json"} files written
by \code{\link[=write_config]{write_config()}}. \code{write_config()} returns a
tibble that can be used as an input (see \code{x}). It serves the details to
submit single or multiple (dependent/subsequent) model simulations.
}
\details{
A \link[tibble]{tibble} for \code{x} that has been generated by
\code{\link[=write_config]{write_config()}} and can look like the following examples can
supplied:\tabular{l}{
   \strong{sim_name} \cr
   scen1_spinup \cr
   scen2_transient \cr
}


To perform subsequent or rather dependent simulations the optional run
parameter \code{"dependency"} needs to be provided within the initial
\link[tibble]{tibble} supplied as \code{param} to \code{\link[=write_config]{write_config()}}.\tabular{ll}{
   \strong{sim_name} \tab \strong{dependency} \cr
   scen1_spinup \tab NA \cr
   scen2_transient \tab scen1 _spinup \cr
}


To use different SLURM settings for each run the optional SLURM options
\code{"sclass"}, \code{"ntask"}, \code{"wtime"} or \code{"blocking"} can also be supplied to the
initial \link[tibble]{tibble} supplied as \code{param} to
\code{\link[=write_config]{write_config()}}. These overwrite the (default) SLURM
arguments (\code{sclass}, \code{ntask}, \code{wtime} or \code{blocking}) supplied to
\code{submit_lpjml}.\tabular{llr}{
   \strong{sim_name} \tab \strong{dependency} \tab \strong{wtime} \cr
   scen1_spinup \tab NA \tab "8:00:00" \cr
   scen2_transient \tab scen1 _spinup \tab "2:00:00" \cr
}


As a shortcut it is also possible to provide the config file
\code{"config_*.json"} as a character string or multiple config files as a
character string vector directly as the \code{x} argument to \code{submit_lpjml}. \cr
With this approach, run parameters or SLURM options cannot be taken into
account. \cr
}
\examples{

\dontrun{
library(tibble)

model_path <- "./LPJmL_internal"
sim_path <-"./my_runs"


# Basic usage
my_params <- tibble(
 sim_name = c("scen1", "scen2"),
 random_seed = as.integer(c(42, 404)),
  `pftpar[[1]]$name` = c("first_tree", NA),
  `param$k_temp` = c(NA, 0.03),
 new_phenology = c(TRUE, FALSE)
)

config_details <- write_config(my_params, model_path, sim_path)

 run_details <- submit_lpjml(
  x = config_details,
  model_path = model_path,
  sim_path = sim_path
)

run_details
#   sim_name      job_id   status
#   <chr>           <int>  <chr>
# 1 scen1        21235215  submitted
# 2 scen2        21235216  submitted


# With run parameter dependency and SLURM option wtime being
#   set (also less other parameters than in previous example)
my_params <- tibble(
  sim_name = c("scen1", "scen2"),
  random_seed = as.integer(c(42, 404)),
  dependency = c(NA, "scen1_spinup"),
  wtime = c("8:00:00", "4:00:00"),
)

config_details2 <- write_config(my_params2, model_path, sim_path)

run_details2 <- submit_lpjml(config_details2, model_path, sim_path)

run_details2
#   sim_name        order dependency   wtime   type       job_id   status
#   <chr>           <dbl> <chr>        <chr>   <chr>      <chr>    <chr>
# 1 scen1_spinup        1 NA           8:00:00 simulation 22910240 submitted
# 2 scen1_transient     2 scen1_spinup 4:00:00 simulation 22910241 submitted


# Same but by using the pipe operator
library(magrittr)

run_details <- tibble(
  sim_name = c("scen1_spinup", "scen1_transient"),
  random_seed = as.integer(c(1, 42)),
  dependency = c(NA, "scen1_spinup"),
  wtime = c("8:00:00", "4:00:00"),
) \%>\%
  write_config(model_path, sim_path) \%>\%
  submit_lpjml(model_path, sim_path)


# Shortcut approach
run_details <- submit_lpjml(
  x = "./config_scen1_transient.json",
  model_path = model_path,
  sim_path = sim_path
)

run_details <- submit_lpjml(
  c("./config_scen1_spinup.json", "./config_scen1_transient.json"),
  model_path,
  sim_path
)

}

}
