% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/xval.R
\name{lol.xval.optimal_dimselect}
\alias{lol.xval.optimal_dimselect}
\title{Optimal Cross-Validated Number of Embedding Dimensions}
\usage{
lol.xval.optimal_dimselect(
  X,
  Y,
  rs,
  alg,
  sets = NULL,
  alg.dimname = "r",
  alg.opts = list(),
  alg.embedding = "A",
  alg.structured = TRUE,
  classifier = lda,
  classifier.opts = list(),
  classifier.return = "class",
  k = "loo",
  rank.low = FALSE,
  ...
)
}
\arguments{
\item{X}{\code{[n, d]} the data with \code{n} samples in \code{d} dimensions.}

\item{Y}{\code{[n]} the labels of the samples with \code{K} unique labels. Defaults to \code{NaN}.#' @param alg.opts any extraneous options to be passed to the classifier function, as a list. Defaults to an empty list. For example, this could be the embedding dimensionality to investigate.}

\item{rs}{\code{[r.n]} the embedding dimensions to investigate over, where \code{max(rs) <= d}.}

\item{alg}{the algorithm to use for embedding. Should be a function that accepts inputs \code{X} and \code{Y} and embedding dimension \code{r} if \code{alg} is supervised, or just \code{X} and embedding dimension \code{r} if \code{alg} is unsupervised.This algorithm should return a list containing a matrix that embeds from {d} to {r < d} dimensions.}

\item{sets}{a user-defined cross-validation set. Defaults to \code{NULL}.
\itemize{
\item{\code{is.null(sets)} randomly partition the inputs \code{X} and \code{Y} into training and testing sets.}
\item{\code{!is.null(sets)} use a user-defined partitioning of the inputs \code{X} and \code{Y} into training and testing sets. Should be in the format of the outputs from \code{\link{lol.xval.split}}. That is, a \code{list} with each element containing \code{X.train}, an \code{[n-k][d]} subset of data to test on, \code{Y.train}, an \code{[n-k]} subset of class labels for \code{X.train}; \code{X.test}, an \code{[n-k][d]} subset of data to test the model on, \code{Y.train}, an \code{[k]} subset of class labels for \code{X.test}.}
}}

\item{alg.dimname}{the name of the parameter accepted by \code{alg} for indicating the embedding dimensionality desired. Defaults to \code{r}.}

\item{alg.opts}{the hyper-parameter options to pass to your algorithm as a keyworded list. Defaults to \code{list()}, or no hyper-parameters. This should not include the number of embedding dimensions, \code{r}, which are passed separately in the \code{rs} vector.}

\item{alg.embedding}{the attribute returned by \code{alg} containing the embedding matrix. Defaults to assuming that \code{alg} returns an embgedding matrix as \code{"A"}.
\itemize{
\item \code{!is.nan(alg.embedding)} Assumes that \code{alg} will return a list containing an attribute, \code{alg.embedding}, a \code{[d, r]} matrix that embeds \code{[n, d]} data from \code{[d]} to \code{[r < d]} dimensions.
\item \code{is.nan(alg.embedding)} Assumes that \code{alg} returns a \code{[d, r]} matrix that embeds \code{[n, d]} data from \code{[d]} to \code{[r < d]} dimensions.
}}

\item{alg.structured}{a boolean to indicate whether the embedding matrix is structured. Provides performance increase by not having to compute the embedding matrix \code{xv} times if unnecessary. Defaults to \code{TRUE}.
\itemize{
\item \code{TRUE} assumes that if \code{Ar: R^d -> R^r} embeds from \code{d} to \code{r} dimensions and \code{Aq: R^d -> R^q} from \code{d} to \code{q > r} dimensions, that \code{Aq[, 1:r] == Ar},
\item \code{TRUE} assumes that if \code{Ar: R^d -> R^r} embeds from \code{d} to \code{r} dimensions and \code{Aq: R^d -> R^q} from \code{d} to \code{q > r} dimensions, that \code{Aq[, 1:r] != Ar},
}}

\item{classifier}{the classifier to use for assessing performance. The classifier should accept \code{X}, a \code{[n, d]} array as the first input, and \code{Y}, a \code{[n]} array of labels, as the first 2 arguments. The class should implement a predict function, \code{predict.classifier}, that is compatible with the \code{stats::predict} \code{S3} method. Defaults to \code{MASS::lda}.}

\item{classifier.opts}{any extraneous options to be passed to the classifier function, as a list. Defaults to an empty list.}

\item{classifier.return}{if the return type is a list, \code{class} encodes the attribute containing the prediction labels from \code{stats::predict}. Defaults to the return type of \code{MASS::lda}, \code{class}.
\itemize{
\item{\code{!is.nan(classifier.return)} Assumes that \code{predict.classifier} will return a list containing an attribute, \code{classifier.return}, that encodes the predicted labels.}
\item{\code{is.nan(classifier.return)} Assumes that \code{predict.classifer} returns a \code{[n]} vector/array containing the prediction labels for \code{[n, d]} inputs.}
}}

\item{k}{the cross-validated method to perform. Defaults to \code{'loo'}. If \code{sets} is provided, this option is ignored. See \code{\link{lol.xval.split}} for details.
\itemize{
\item{\code{'loo'} Leave-one-out cross validation}
\item{\code{isinteger(k)}  perform \code{k}-fold cross-validation with \code{k} as the number of folds.}
}}

\item{rank.low}{whether to force the training set to low-rank. Defaults to \code{FALSE}. If \code{sets} is provided, this option is ignored. See \code{\link{lol.xval.split}} for details.
\itemize{
\item{if \code{rank.low == FALSE}, uses default cross-validation method with standard \code{k}-fold validation. Training sets are \code{k-1} folds, and testing sets are \code{1} fold, where the fold held-out for testing is rotated to ensure no dependence of potential downstream inference in the cross-validated misclassification rates.}
\item{if ]code{rank.low == TRUE}, users cross-validation method with \code{ntrain = min((k-1)/k*n, d)} sample training sets, where \code{d}  is the number of dimensions in \code{X}. This ensures that the training data is always low-rank, \code{ntrain < d + 1}. Note that the resulting training sets may have \code{ntrain < (k-1)/k*n}, but the resulting testing sets will always be properly rotated \code{ntest = n/k} to ensure no dependencies in fold-wise testing.}
}}

\item{...}{trailing args.}
}
\value{
Returns a list containing:
\item{\code{folds.data}}{the results, as a data-frame, of the per-fold classification accuracy.}
\item{\code{foldmeans.data}}{the results, as a data-frame, of the average classification accuracy for each \code{r}.}
\item{\code{optimal.lhat}}{the classification error of the optimal \code{r}}.
\item{\code{optimal.r}}{the optimal number of embedding dimensions from \code{rs}}.
\item{\code{model}}{the model trained on all of the data at the optimal number of embedding dimensions.}
\item{\code{classifier}}{the classifier trained on all of the data at the optimal number of embedding dimensions.}
}
\description{
A function for performing leave-one-out cross-validation for a given embedding model, that allows users to determine the optimal number of embedding dimensions for
their algorithm-of-choice. This function produces fold-wise cross-validated misclassification rates for standard embedding techniques across a specified selection of
embedding dimensions. Optimal embedding dimension is selected as the dimension with the lowest average misclassification rate across all folds.
Users can optionally specify custom embedding techniques with proper configuration of \code{alg.*} parameters and hyperparameters.
Optional classifiers implementing the S3 \code{predict} function can be used for classification, with hyperparameters to classifiers for
determining misclassification rate specified in \code{classifier.*}.
}
\section{Details}{

For more details see the help vignette:
\code{vignette("xval", package = "lolR")}

For  extending cross-validation techniques shown here to arbitrary embedding algorithms, see the vignette:
\code{vignette("extend_embedding", package = "lolR")}

For  extending cross-validation techniques shown here to arbitrary classification algorithms, see the vignette:
\code{vignette("extend_classification", package = "lolR")}
}

\examples{
# train model and analyze with loo validation using lda classifier
library(lolR)
data <- lol.sims.rtrunk(n=200, d=30)  # 200 examples of 30 dimensions
X <- data$X; Y <- data$Y
# run cross-validation with the nearestCentroid method and
# leave-one-out cross-validation, which returns only
# prediction labels so we specify classifier.return as NaN
xval.fit <- lol.xval.optimal_dimselect(X, Y, rs=c(5, 10, 15), lol.project.lol,
                          classifier=lol.classify.nearestCentroid,
                          classifier.return=NaN, k='loo')

# train model and analyze with 5-fold validation using lda classifier
data <- lol.sims.rtrunk(n=200, d=30)  # 200 examples of 30 dimensions
X <- data$X; Y <- data$Y
xval.fit <- lol.xval.optimal_dimselect(X, Y, rs=c(5, 10, 15), lol.project.lol, k=5)

# pass in existing cross-validation sets
sets <- lol.xval.split(X, Y, k=2)
xval.fit <- lol.xval.optimal_dimselect(X, Y, rs=c(5, 10, 15), lol.project.lol, sets=sets)
}
\author{
Eric Bridgeford
}
