% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/kre_adaptive.R
\name{weights_ES2012}
\alias{weights_ES2012}
\title{Weights \eqn{W_{ni}} of Eichner & Stute (2012)}
\usage{
weights_ES2012(sigma, xXh, thetaXh, K, h)
}
\arguments{
\item{sigma}{Numeric vector \eqn{(\sigma_1, \ldots, \sigma_s)} with
\eqn{s \ge 1} with values of the scale parameter \eqn{\sigma}.}

\item{xXh}{Numeric vector expecting the pre-computed h-scaled differences
\eqn{(x - X_1)/h}, \ldots, \eqn{(x - X_n)/h} where \eqn{x} is the
single (!) location for which the weights are to be computed,
the \eqn{X_i}'s are the data values, and \eqn{h} is the numeric
bandwidth scalar.}

\item{thetaXh}{Numeric vector expecting the pre-computed h-scaled differences
\eqn{(\theta - X_1)/h}, \ldots, \eqn{(\theta - X_n)/h} where
\eqn{\theta} is the numeric scalar location parameter, and the
\eqn{X_i}'s and \eqn{h} are as in \code{xXh}.}

\item{K}{A kernel function (with vectorized in- & output) to be used for the
estimator.}

\item{h}{Numeric scalar for bandwidth \eqn{h} (as ``contained'' in
\code{thetaXh} and \code{xXh}).}
}
\value{
If \code{length(sigma)} > 1 a numeric matrix of the dimension
        \code{length(sigma)} by \code{length(xXh)} with elements
        \eqn{(W_{ni}(x; h, \theta, \sigma_r))} for \eqn{r = 1, \ldots,}
        \code{length(sigma)} and \eqn{i = 1, \ldots,} \code{length(xXh)};
        otherwise a numeric vector of the same length as \code{xXh}.
}
\description{
Function, vectorized in its first argument \code{sigma}, to compute the
``updated'' weights \eqn{W_{ni}} in eq. (2.1) of Eichner & Stute (2012) for
the kernel adjusted regression estimator.
}
\details{
Note that it is not immediately obvious that \eqn{W_{ni}} in eq. (2.1) of
Eichner & Stute (2012) is a function of \eqn{\sigma}. In fact, \eqn{W_{ni}
= W_{ni}(x; h, \theta, \sigma)} as can be seen on p. 2542 ibid. The
computational version implemented here, however, is given in (15.19) of
Eichner (2017). Pre-computed \eqn{(x - X_i)/h} and \eqn{(\theta - X_i)/h},
\eqn{i = 1, \ldots, n} are expected for efficiency reasons (and are
currently prepared in function \code{\link{kare}}).
}
\examples{
require(stats)

 # Regression function:
m <- function(x, x1 = 0, x2 = 8, a = 0.01, b = 0) {
 a * (x - x1) * (x - x2)^3 + b
}
 # Note: For a few details on m() see examples in ?nadwat.

n <- 100       # Sample size.
set.seed(42)   # To guarantee reproducibility.
X <- runif(n, min = -3, max = 15)      # X_1, ..., X_n   # Design.
Y <- m(X) + rnorm(length(X), sd = 5)   # Y_1, ..., Y_n   # Response.

h <- n^(-1/5)
Sigma <- seq(0.01, 10, length = 51)   # sigma-grid for minimization.
x0 <- 5   # Location at which the estimator of m should be computed.

 # Weights (W_{ni}(x; \\sigma_r))_{1<=r<=length(Sigma), 1<=i<=n} for
 # Var_n(sigma) and Bias_n(sigma) each at x0 on the sigma-grid:
weights_ES2012(sigma = Sigma, xXh = (x0 - X) / h,
  thetaXh = (mean(X) - X) / h, K = dnorm, h = h)

}
\references{
Eichner & Stute (2012) and Eichner (2017): see \code{\link{kader}}.
}
\seealso{
\code{\link{bias_ES2012}} and \code{\link{var_ES2012}} which both
         call this function, and \code{\link{kare}} which currently does
         the pre-computing.
}
