% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/kde_adaptive.R
\name{fnhat_ES2013}
\alias{fnhat_ES2013}
\title{Robust Kernel Density Estimator of Eichner & Stute (2013)}
\usage{
fnhat_ES2013(x, data, K, h, ranktrafo, sigma)
}
\arguments{
\item{x}{Numeric vector with the location(s) at which the density
estimate is to be computed.}

\item{data}{Numeric vector \eqn{(X_1, \ldots, X_n)} of the data from
which the estimate is to be computed. Missing values are not
allowed and entail an error.}

\item{K}{A kernel function to be used for the estimator.}

\item{h}{Numeric scalar for bandwidth \eqn{h}.}

\item{ranktrafo}{A function used for the rank transformation.}

\item{sigma}{Numeric scalar for value of scale parameter \eqn{\sigma}.}
}
\value{
An object with class "density" whose underlying structure is
        a list containing the following components (as described in
        \code{\link[stats]{density}}), so that the \code{print} and
        \code{plot} methods for \code{density}-objects are
        immediately available):
\tabular{ll}{
 \code{x}  \tab the n coordinates of the points where the density is
                estimated. \cr
 \code{y}  \tab the estimated density values from eq. (4) in Eichner & Stute
                (2013). \cr
 \code{bw} \tab the bandwidth used. \cr
 \code{n}  \tab the sample size. (Recall: missing or infinite values are not
                allowed here.) \cr
 \code{call}      \tab the call which produced the result. \cr
 \code{data.name} \tab the deparsed name of the x argument. \cr
 \code{has.na}    \tab logical, for compatibility (always FALSE). \cr\cr
 Additionally: \tab \cr
 \code{ranktrafo} \tab as in Arguments. \cr
 \code{sigma} \tab as in Arguments. \cr
 }
}
\description{
Implementation of eq. (4) in Eichner & Stute (2013) for a given and fixed
scalar \eqn{\sigma}, for rank transformation function \eqn{J} (and, of
course, for fixed and given location(s) in \eqn{x}, data \eqn{(X_1, \ldots,
X_n)}, a kernel function \eqn{K}, and a bandwidth \eqn{h}).
}
\details{
The formula upon which the computational version implemented here is based
is given in eq. (15.9) of Eichner (2017). This function does mainly only a
simple preparatory computation and then calls \code{\link{compute_fnhat}}
which does the actual work.
}
\examples{
require(stats);   require(grDevices);   require(datasets)

 # Simulated N(0,1)-data and one sigma-value
set.seed(2016);     n <- 100;     d <- rnorm(n)
xgrid <- seq(-4, 4, by = 0.1)
(fit <- fnhat_ES2013(x = xgrid, data = d, K = dnorm, h = n^(-1/5),
  ranktrafo = J2, sigma = 1) )
\donttest{
plot(fit, ylim = range(0, dnorm(0), fit$y), col = "blue")
curve(dnorm, add = TRUE);   rug(d, col = "red")
legend("topleft", lty = 1, col = c("blue", "black", "red"),
  legend = expression(hat(f)[n], phi, "data"))
}
\donttest{
 # The same data, but several sigma-values
sigmas <- seq(1, 4, length = 4)
(fit <- lapply(sigmas, function(sig)
  fnhat_ES2013(x = xgrid, data = d, K = dnorm, h = n^(-1/5),
    ranktrafo = J2, sigma = sig)) )

ymat <- sapply(fit, "[[", "y")
matplot(x = xgrid, y = ymat, type = "l", lty = 1, col = 2 + seq(sigmas),
  ylim = range(0, dnorm(0), ymat), main = "", xlab = "", ylab = "Density")
curve(dnorm, add = TRUE);   rug(d, col = "red")
legend("topleft", lty = 1, col = c("black", "red", NA), bty = "n",
  legend = expression(phi, "data", hat(f)[n]~"in other colors"))
}
\donttest{
 # Old-Faithful-eruptions-data and several sigma-values
d <- faithful$eruptions;     n <- length(d);     er <- extendrange(d)
xgrid <- seq(er[1], er[2], by = 0.1);    sigmas <- seq(1, 4, length = 4)
(fit <- lapply(sigmas, function(sig)
   fnhat_ES2013(x = xgrid, data = d, K = dnorm, h = n^(-1/5),
     ranktrafo = J2, sigma = sig)) )

ymat <- sapply(fit, "[[", "y");     dfit <- density(d, bw = "sj")
plot(dfit, ylim = range(0, dfit$y, ymat), main = "", xlab = "")
rug(d, col = "red")
matlines(x = xgrid, y = ymat, lty = 1, col = 2 + seq(sigmas))
legend("top", lty = 1, col = c("black", "red", NA), bty = "n",
  legend = expression("R's est.", "data", hat(f)[n]~"in other colors"))
}
}
\references{
Eichner & Stute (2013) and Eichner (2017): see \link{kader}.
}
\seealso{
\code{\link{fnhat_SS2011}}.
}
