# WARNING - Generated by {fusen} from dev/flat_clean_fusen_files.Rmd: do not edit by hand

#' Show in a table files that are not already registered in the yaml config file
#'
#' If user start their package without 'fusen' or with version < 0.4, they need to create the config file, with already existing functions.
#'
#' @param path Path to package to check for not registered files
#' @param guess Logical. Guess if the file was inflated by a specific flat file
#' @param to_csv Logical. Whether to store along the config file, the outputs in a csv for the user to clean it manually
#' @return Path to csv file if `to_csv` is TRUE. `dput()` of the dataframe otherwise.
#' @importFrom utils write.csv
#'
#' @export
#' @examples
#' \dontrun{
#' # Run this on the current package in development
#' out_csv <- check_not_registered_files()
#' out_csv
#' }
#'
#' # Or you can try on the reproducible example
#' dummypackage <- tempfile("clean")
#' dir.create(dummypackage)
#'
#' # {fusen} steps
#' fill_description(pkg = dummypackage, fields = list(Title = "Dummy Package"))
#' dev_file <- suppressMessages(add_flat_template(pkg = dummypackage, overwrite = TRUE, open = FALSE))
#' flat_file <- dev_file[grepl("flat_", dev_file)]
#' # Inflate once
#' usethis::with_project(dummypackage, {
#'   suppressMessages(
#'     inflate(
#'       pkg = dummypackage, flat_file = flat_file,
#'       vignette_name = "Get started", check = FALSE,
#'       open_vignette = FALSE
#'     )
#'   )
#'
#'   # Add a not registered file to the package
#'   cat("# test R file", file = file.path(dummypackage, "R", "to_keep.R"))
#'
#'   # Use the fonction to check the list of files
#'   out_csv <- check_not_registered_files(dummypackage)
#'   out_csv
#'   # Open the csv to modify manually the list of files in the csv created
#'   # Indeed, the "to_keep.R" file will be registered as
#'   # "No existing source path found."
#'   # Manually change this line with "keep"
#'   content_csv <- read.csv(out_csv, stringsAsFactors = FALSE)
#'   # Here I change the line to simulate what you manually did above
#'   content_csv[content_csv[["path"]] == "R/to_keep.R", "origin"] <- "keep"
#'   write.csv(content_csv, out_csv)
#'
#'   out_config <- df_to_config(df_files = out_csv)
#'   out_config
#'   # Open the out_config file to see what's going on
#'   yaml::read_yaml(out_config)
#' })
#' unlink(dummypackage, recursive = TRUE)
check_not_registered_files <- function(path = ".", guess = TRUE, to_csv = TRUE) {
  path <- normalizePath(path, winslash = "/")

  all_r <- list.files(file.path(path, "R"), pattern = "[.]R$|[.]r$", full.names = TRUE)
  all_test <- list.files(file.path(path, "tests", "testthat"), pattern = "[.]R$|[.]r$", full.names = TRUE)
  all_vignette <- list.files(file.path(path, "vignettes"), pattern = "[.]Rmd$|[.]rmd$", full.names = TRUE)

  res <- data.frame(
    type = c(
      rep("R", length(all_r)),
      rep("test", length(all_test)),
      rep("vignette", length(all_vignette))
    ),
    path = c(
      all_r, all_test, all_vignette
    ),
    stringsAsFactors = FALSE
  )

  if (nrow(res) == 0) {
    message("There are no files in the package. Did you inflate it once?")
    return(NULL)
  }

  config_file <- getOption("fusen.config_file", default = "dev/config_fusen.yaml")

  if (file.exists(config_file)) {
    # Read config file, and remove those already there
    config_list <- yaml::read_yaml(config_file)
    config_list_path <- normalizePath(get_list_paths(config_list), mustWork = FALSE, winslash = "/")
    res_existing <- res[res$path %in% config_list_path, ]
    res_new <- res[!res$path %in% config_list_path, ]
  } else {
    res_existing <- res[FALSE, ]
    res_new <- res
  }

  if (nrow(res_new) == 0) {
    cli::cli_alert_info("There are no unregistered files")
    return(NULL)
  }

  # Guess those not in config_file
  # All files without path, are changed for "keep"
  if (isTRUE(guess)) {
    # List all files
    res_new$origin <- unlist(lapply(res_new$path, guess_flat_origin))
  } else {
    res_new$origin <- "keep"
  }

  # TODO go back to relative path
  res_new$path <- gsub(paste0(getwd(), "/"), "", normalize_path_winslash(res_new$path, mustWork = TRUE))

  # config_file may not exist already
  csv_file <- file.path(
    gsub(paste0(getwd(), "/"), "", dirname(normalize_path_winslash(config_file, mustWork = FALSE))), "config_not_registered.csv"
  )

  # Save for manual modification
  if (isTRUE(to_csv)) {
    write.csv(res_new, csv_file, row.names = FALSE)
    cli::cli_alert_info(paste(
      "Wrote not registered files in: ", csv_file,
      "\nKeep only those necessary and run `df_to_config()` on the csv file."
    ))

    return(csv_file)
  } else {
    # return a `dput()` to allow to add to `df_to_config()`
    # dput()
    return(dput(res_new))
  }
}

#' Guess flat file of origin of a script
#'
#' @param path Path of the script to test
#' @noRd
guess_flat_origin <- function(path) {
  lines <- readLines(path)

  guess_path <- sub(
    ".* from\\s*(/){0,1}(.+[.].{1}md).*", "\\2",
    lines[grep("(G|g)enerated by \\{fusen\\} from", lines)][1]
  )

  guess_path <- normalizePath(guess_path, mustWork = FALSE, winslash = "/")
  if (file.exists(guess_path)) {
    guess_path <- gsub(paste0(getwd(), "/"), "", normalizePath(guess_path, mustWork = FALSE, winslash = "/"))
    return(guess_path)
  } else {
    return("No existing source path found. Write 'keep', the full path to the flat file source, or delete this line.")
  }
}

#' Get all paths from a config list, except some
#' @param config_list List as read from config.yaml file
#' @noRd
get_list_paths <- function(config_list) {
  unlist(lapply(config_list, function(x) {
    if (x$path == "keep") x$path <- NULL
    x$state <- NULL
    x
  }))
}

#' Title
#'
#' Description
#'
#' @return Used for side effect. Delete unregistered files.
#'
#' @export
#' @examples
#' clean_fusen_files()
clean_fusen_files <- function() {
  message("This does nothing for now")
}

#' Add a tibble of files and types to the 'fusen' config file
#' @param df_files A dataframe with 'type' and 'path' columns
#' or a csv file path as issued from `[check_not_registered_files()]`
#' or nothing (and it will take the csv file in "dev/")
#' @param flat_file_path Character. Usually `"keep"` or the name of the origin flat file, although inflating the flat file should have the same result.
#' @param state Character. Whether if the flat file is `active` or `deprecated`.
#' @param force Logical. Whether to force writing the configuration file even is some files do not exist.
#' @param clean Logical. Delete list associated a specific flat file before updating the whole list. Default is set to TRUE during `inflate()` of a specific flat fil, as the list should only contain files created during the inflate. This parameter is set to FALSE with `register_to_config()` so that it can be run twice on the package when migrating from an old version of {fusen}. This could be set to FALSE with a direct use of `df_to_config()` too.
#'
#' @importFrom stats setNames
#' @importFrom utils read.csv
#'
#' @return Config file path.
#' Side effect: create a yaml config file.
#' @export
#'
#' @details
#' - Only types: "R", "test", "vignette" are accepted.
#' - All `path` should exist.
#' - Change path to config file with `options(fusen.config_file = "dev/config_fusen.yaml")`
#'
#' @examples
#' # Add your own list of files to "keep",
#' # if they are not in a flat file.
#' # Otherwise, they may be deleted with your next `inflate()`
#' my_files_to_protect <- tibble::tribble(
#'   ~type, ~path,
#'   "R", "R/zaza.R",
#'   "R", "R/zozo.R",
#'   "test", "tests/testthat/test-zaza.R",
#'   "vignette", "vignettes/my-zaza-vignette.Rmd"
#' )
#'
#' \dontrun{
#' df_to_config(my_files_to_protect)
#' }
df_to_config <- function(df_files, flat_file_path = "keep", state = c("active", "deprecated"), force = FALSE, clean = TRUE) {
  config_file <- getOption("fusen.config_file", default = "dev/config_fusen.yaml")
  state <- match.arg(state, several.ok = FALSE)

  # User entry verifications
  if (missing(df_files)) {
    df_files <- file.path(dirname(config_file), "config_not_registered.csv")
  }

  if (!is.data.frame(df_files) && file.exists(df_files)) {
    df_files <- read.csv(df_files, stringsAsFactors = FALSE)
  } else if (!is.data.frame(df_files) && !file.exists(df_files)) {
    stop("'", df_files, "' does not exist. You can run `check_not_registered_files()` before.")
  }

  # Then if is.data.frame(df_files)
  if (!all(c("type", "path") %in% names(df_files))) {
    stop("df_files should contains two columns named: 'type' and 'path'")
  }



  if (!"origin" %in% names(df_files)) {
    df_files[["origin"]] <- flat_file_path
  } else {
    # check origin exists
    df_files_origin <- df_files[["origin"]]
    df_files_origin <- df_files_origin[df_files_origin != "keep"]
    all_exists <- file.exists(df_files_origin)
    if (!all(all_exists)) {
      msg <- paste(
        "Some 'origin' in df_files do not exist:",
        paste(
          paste0(
            "row ", which(!all_exists), ": ",
            df_files[["origin"]][!all_exists]
          ),
          collapse = ", "
        )
      )
      if (isTRUE(force)) {
        cli::cli_alert_warning(
          paste(msg, "\nHowever, you forced to write it in the yaml file with `force = TRUE`.")
        )
      } else {
        stop(msg)
      }
    }
  }

  if (!all(grepl("^R$|^r$|^test$|^tests$|^vignette$|^vignettes$", df_files[["type"]]))) {
    stop("Only types 'R', 'test' or 'vignette' are allowed")
  }
  all_exists <- file.exists(df_files[["path"]])
  if (!all(all_exists)) {
    msg <- paste(
      "Some 'path' in df_files do not exist:",
      paste(
        paste0(
          "row ", which(!all_exists), "- ",
          df_files[["type"]][!all_exists], ": ",
          df_files[["path"]][!all_exists]
        ),
        collapse = ", "
      )
    )

    if (isTRUE(force)) {
      cli::cli_alert_warning(
        paste(msg, "\nHowever, you forced to write it in the yaml file with `force = TRUE`.")
      )
    } else {
      stop(msg)
    }
  }

  # Remove common part between config_file and all path
  # to get relative paths to project

  # TODO - When path does not exists, normalizePath does not correctly
  # use path.expand.
  # All path should exists. It is tested above.
  df_files$path <- gsub(
    paste0(normalize_path_winslash("."), "/"),
    "",
    normalize_path_winslash(df_files$path, mustWork = TRUE)
  )

  # All origin path should exist, if not "keep"
  df_files$origin[df_files$origin != "keep"] <- gsub(
    paste0(normalize_path_winslash("."), "/"),
    "",
    normalize_path_winslash(df_files$origin[df_files$origin != "keep"], mustWork = TRUE)
  )


  if (any(duplicated(df_files$path))) {
    msg <- paste("Some paths appear multiple times in df_files. Please remove duplicated rows: ", paste(unique(df_files$path[duplicated(df_files$path)]), collapse = ", "))

    if (isTRUE(force)) {
      cli::cli_alert_warning(
        paste(msg, "\nHowever, you forced to write it in the yaml file with `force = TRUE`.")
      )
    } else {
      stop(msg)
    }
  }

  if (file.exists(config_file)) {
    complete_yaml <- yaml::read_yaml(config_file)

    yaml_paths <- get_list_paths(complete_yaml)
    all_exists <- file.exists(yaml_paths)
    if (!all(all_exists)) {
      msg <- paste(
        "Some 'path' in config_file do not exist:",
        paste(
          yaml_paths[!all_exists],
          collapse = ", "
        )
      )
      if (isTRUE(force)) {
        cli::cli_alert_warning(
          paste(msg, "However, you forced to write it in the yaml file with `force = TRUE`.")
        )
      } else {
        stop(msg)
      }
    }
  } else {
    complete_yaml <- list()
  }

  each_flat_file_path <- unique(df_files[["origin"]])
  if (length(state) != length(each_flat_file_path)) {
    state <- rep(state, length.out = length(each_flat_file_path))
  }
  all_groups_list <- lapply(
    seq_along(each_flat_file_path),
    function(x) {
      update_one_group_yaml(
        df_files, complete_yaml,
        each_flat_file_path[x],
        state = state[x],
        clean = clean
      )
    }
  ) %>%
    setNames(basename(each_flat_file_path))

  all_modified <- names(complete_yaml)[names(complete_yaml) %in% names(all_groups_list)]

  if (length(all_modified) != 0) {
    cli::cli_alert_info(
      paste0(
        "Some files group already existed and were overwritten: ",
        paste(all_modified, collapse = ", ")
      )
    )
    complete_yaml[all_modified] <- NULL
  }

  # Combine with complete_yaml
  complete_yaml <- c(complete_yaml, all_groups_list)
  complete_yaml <- complete_yaml[sort(names(complete_yaml))]

  # Get duplicated paths after update
  yaml_paths <- unlist(complete_yaml)
  # Except info on state
  yaml_paths <- yaml_paths[!grepl("[.]state$", names(yaml_paths))]

  if (any(duplicated(yaml_paths))) {
    which_file <- files_list_to_vector(
      lapply(complete_yaml, files_list_to_vector)
    )

    if (isTRUE(force)) {
      cli::cli_alert_warning("Some paths appear multiple times in the config file. The yaml was forced with `force = TRUE`. Please verify the source of these scripts in the existing yaml file or in the df_files provided:\n", paste(which_file[duplicated(yaml_paths)], collapse = ",\n"))
    } else {
      stop("Some paths would appear multiple times in the future config file. The yaml has not been created. Please verify the source of these scripts in the existing yaml file or in the df_files provided:\n", paste(which_file[duplicated(yaml_paths)], collapse = ",\n"), "\nOr use `force = TRUE`")
    }
  }

  yaml::write_yaml(complete_yaml, file = config_file)

  return(config_file)
}

#' Extract name of file along with type to inform the user
#' @param list_of_files A named list of files
#' @noRd
files_list_to_vector <- function(list_of_files) {
  lapply(seq_along(list_of_files), function(x) {
    if (length(list_of_files[[x]]) != 0) {
      paste(names(list_of_files[x]), list_of_files[[x]], sep = ": ")
    }
  }) %>% unlist()
}

#' Update one group in the complete yaml as list
#' @param complete_yaml The list as output of config_yaml file
#' @param flat_file_path The group to update
#' @param state Character. "active" or "deprecated"
#' @param clean Logical. See `df_to_config()`. Delete list associated a specific flat file before updating the whole list. Default is set to TRUE during `inflate()` of a specific flat fil, as the list should only contain files created during the inflate. This parameter is set to FALSE with `register_to_config()` so that it can be run twice on the package when migrating from an old version of {fusen}. This could be set to FALSE with a direct use of `df_to_config()` too.
#' @noRd
update_one_group_yaml <- function(df_files, complete_yaml, flat_file_path, state = c("active", "deprecated"), clean = TRUE) {
  state <- match.arg(state, several.ok = FALSE)
  all_keep_before <- complete_yaml[[basename(flat_file_path)]]

  # Only files from df_files will be listed
  df_files_filtered <- df_files[df_files[["origin"]] == flat_file_path, ]

  # All already in the list will be deleted except if clean is FALSE
  if (isTRUE(clean)) {
    this_group_list <- list(
      path = flat_file_path,
      state = state,
      R = c(df_files_filtered[["path"]][
        grepl("^R$|^r$", df_files_filtered[["type"]])
      ]),
      tests = c(df_files_filtered[["path"]][
        grepl("^test$|^tests$", df_files_filtered[["type"]])
      ]),
      vignettes = c(df_files_filtered[["path"]][
        grepl("^vignette$|^vignettes$", df_files_filtered[["type"]])
      ])
    )
  } else {
    this_group_list <- list(
      path = flat_file_path,
      state = state,
      R = c(
        # new ones
        df_files_filtered[["path"]][
          grepl("^R$|^r$", df_files_filtered[["type"]])
        ],
        # previous ones
        unlist(all_keep_before[["R"]])
      ),
      tests = c(
        # new ones
        df_files_filtered[["path"]][
          grepl("^test$|^tests$", df_files_filtered[["type"]])
        ],
        # previous ones
        unlist(all_keep_before[["tests"]])
      ),
      vignettes = c(
        # new ones
        df_files_filtered[["path"]][grepl("^vignette$|^vignettes$", df_files_filtered[["type"]])],
        # previous ones
        unlist(all_keep_before[["vignettes"]])
      )
    )
  }

  # Those removed
  those_removed <- setdiff(
    all_keep_before,
    this_group_list
  )
  those_removed_vec <- files_list_to_vector(those_removed)
  those_added <- setdiff(
    this_group_list,
    all_keep_before
  )
  those_added_vec <- files_list_to_vector(those_added)

  if (!is.null(those_removed_vec) || length(those_removed_vec) != 0) {
    silent <- lapply(paste(those_removed_vec, "was removed from the config file"), cli::cli_alert_warning)
  }
  if (!is.null(those_added_vec) || length(those_added_vec) != 0) {
    silent <- lapply(paste(those_added_vec, "was added to the config file"), cli::cli_alert_success)
  }

  return(this_group_list)
}

#' Include all existing package files in the config file
#'
#' Helps transition from non-fusen packages
#'
#' @param pkg Path to the package from which to add file to configuration file
#' @inheritParams df_to_config
#' @return Path to configuration file
#'
#' @export
#' @examples
#' \dontrun{
#' # Usually run this one inside the current project
#' # Note: running this will write "dev/config_fusen.yaml" in your working directory
#' register_all_to_config()
#' }
#'
#' # Or you can try on the reproducible example
#' dummypackage <- tempfile("register")
#' dir.create(dummypackage)
#'
#' # {fusen} steps
#' fill_description(pkg = dummypackage, fields = list(Title = "Dummy Package"))
#' dev_file <- suppressMessages(add_flat_template(pkg = dummypackage, overwrite = TRUE, open = FALSE))
#' flat_file <- dev_file[grepl("flat_", dev_file)]
#' # Inflate once
#' usethis::with_project(dummypackage, {
#'   suppressMessages(
#'     inflate(
#'       pkg = dummypackage, flat_file = flat_file,
#'       vignette_name = "Get started", check = FALSE,
#'       open_vignette = FALSE
#'     )
#'   )
#'
#'   out_path <- register_all_to_config(dummypackage)
#'
#'   # Look at the output
#'   yaml::read_yaml(out_path)
#' })
register_all_to_config <- function(pkg = ".", clean = FALSE) {
  # Use the fonction to check the list of files
  out_df <- check_not_registered_files(pkg, to_csv = FALSE)

  if (is.null(out_df)) {
    message("There is no file to register or everything was already registered")
    return(getOption("fusen.config_file", default = "dev/config_fusen.yaml"))
  }

  w.keep <- grep("No existing source path found", out_df[["origin"]])
  out_df[["origin"]][w.keep] <- "keep"
  out_config <- df_to_config(df_files = out_df, clean = clean)
  # rstudioapi::navigateToFile(file.path(pkg, "dev", "config_fusen.yaml"))
  return(invisible(out_config))
}
