% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/flist.R
\name{as.flist}
\alias{as.flist}
\title{Convert to a formatting list}
\usage{
as.flist(x, type = "column", lookup = NULL, simplify = TRUE)
}
\arguments{
\item{x}{Object to convert.}

\item{type}{The type of formatting list.  Valid values are 'row' or 'column'.
The default value is 'column'.}

\item{lookup}{A lookup vector.  Used for looking up the format from 
the formatting list.  This parameter is only used for 'row' type 
formatting lists.}

\item{simplify}{Whether to simplify the results to a vector.  Valid values 
are TRUE or FALSE.  Default is TRUE.  If the value is set to FALSE, the 
return type will be a list.}
}
\value{
A formatting list object.
}
\description{
Converts an object to a formatting list.  All
other parameters are the same as the \code{flist} function.
}
\details{
To apply more than one formatting object to a vector, use a formatting
list.  There are two types of formatting list: column and row.  The column
type formatting lists applies all formats to all values in the
vector.  The row type formatting list can apply a different format to 
each value in the vector.  

Further, there are two styles of row type list: ordered and lookup.  The
ordered style applies each format in the list to the vector values
in the order specified.  The
ordered style will recycle the formats as needed.  The lookup style 
formatting list uses a lookup to determine which format from the list to
apply to a particular value of the vector.  The lookup column values should
correspond to names on the formatting list.  

Examples of column type and row type formatting lists are given below.
}
\examples{
## Example 1: Formatting List - Column Type ##
# Set up data
v1 <- c(Sys.Date(), Sys.Date() + 30, Sys.Date() + 60)

# Create formatting list
fl1 <- flist("\%B", "The month is: \%s")

# Apply formatting list to vector
fapply(v1, fl1)
# [1] "The month is: October"  "The month is: November" "The month is: December"

## Example 2: Formatting List - Row Type ordered ##
# Set up data
# Notice each row has a different data type
l1 <- list("A", 1.263, as.Date("2020-07-21"), 
          "B", 5.8732, as.Date("2020-10-17"))
          
# These formats will be recycled in the order specified           
fl2 <- flist(type = "row",
             c(A = "Label A", B = "Label B"),
             "\%.1f",
             "\%d\%b\%Y")

fapply(l1, fl2)
# [1] "Label A"   "1.3"       "21Jul2020" "Label B"   "5.9"       "17Oct2020"


## Example 3: Formatting List - Row Type with lookup ##
# Set up data
# Notice each row has a different data type
l2 <- list(2841.258, "H", as.Date("2020-06-19"),
           "L", as.Date("2020-04-24"), 1382.8865)
v3 <- c("num", "char", "date", "char", "date", "num")

# Create formatting list
fl3 <- flist(type = "row", lookup = v3,
             num = function(x) format(x, digits = 2, nsmall = 1, 
                                  big.mark=","),
             char = value(condition(x == "H", "High"),
                     condition(x == "L", "Low"),
                     condition(TRUE, "NA")),
             date = "\%d\%b\%Y")

# Apply formatting list to vector, using lookup
fapply(l2, fl3)
# [1] "2,841.3"   "High"      "19Jun2020" "Low"       "24Apr2020" "1,382.9"
}
\seealso{
Other flist: 
\code{\link{as.data.frame.fmt_lst}()},
\code{\link{as.flist.data.frame}()},
\code{\link{as.flist.fcat}()},
\code{\link{as.flist.list}()},
\code{\link{as.flist.tbl_df}()},
\code{\link{flist}()},
\code{\link{is.flist}()},
\code{\link{print.fmt_lst}()}
}
\concept{flist}
