% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/errors.R
\docType{package}
\name{errors-package}
\alias{errors-package}
\title{\pkg{errors}: Uncertainty Propagation for R Vectors}
\description{
Support for measurement errors in R vectors, matrices and arrays: automatic
uncertainty propagation and reporting.
}
\details{
Every measurement has an unknown error associated. Uncertainty is the
acknowledgement of that error: we are aware that our representation of reality
may differ from reality itself. This package provides support for measurement
errors in \R vectors, matrices and arrays. Uncertainty metadata is associated
to quantity values (see \code{\link{errors}}), and this uncertainty is
automatically propagated when you operate with \code{errors} objects (see
\code{\link{groupGeneric.errors}}), or with \code{errors} and numeric objects
(then numeric values are automatically coerced to \code{errors} objects with
no uncertainty).

Correlations between measurements are also supported. In particular, any
operation (e.g., \code{z <- x + y}) results in a correlation between output
and input variables (i.e., \code{z} is correlated to \code{x} and \code{y},
even if there was no correlation between \code{x} and \code{y}). And in
general, the user can establish correlations between any pair of variables
(see \code{\link{correl}}).

This package treats uncertainty as coming from Gaussian and linear sources
(note that, even for non-Gaussian non-linear sources, this is a reasonable
assumption for averages of many measurements), and propagates them using the
first-order Taylor series method for propagation of uncertainty. Although the
above assumptions are valid in a wide range of applications in science and
engineering, the practitioner should evaluate whether they apply for each
particular case.
}
\examples{
## Simultaneous resistance and reactance measurements

# Obtain mean values and uncertainty from measured values
V   <- mean(set_errors(GUM.H.2$V))
I   <- mean(set_errors(GUM.H.2$I))
phi <- mean(set_errors(GUM.H.2$phi))

# Set correlations between variables
correl(V, I)   <- with(GUM.H.2, cor(V, I))
correl(V, phi) <- with(GUM.H.2, cor(V, phi))
correl(I, phi) <- with(GUM.H.2, cor(I, phi))

# Computation of resistance, reactance and impedance values
(R <- (V / I) * cos(phi))
(X <- (V / I) * sin(phi))
(Z <- (V / I))

# Correlations between derived quantities
correl(R, X)
correl(R, Z)
correl(X, Z)

## Calibration of a thermometer

# Least-squares fit for a reference temperature of 20 degC
fit <- lm(bk ~ I(tk - 20), data = GUM.H.3)

# Extract coefficients and set correlation using the covariance matrix
y1 <- set_errors(coef(fit)[1], sqrt(vcov(fit)[1, 1]))
y2 <- set_errors(coef(fit)[2], sqrt(vcov(fit)[2, 2]))
covar(y1, y2) <- vcov(fit)[1, 2]

# Predicted correction for 30 degC
(b.30 <- y1 + y2 * set_errors(30 - 20))

}
\references{
Iñaki Ucar, Edzer Pebesma and Arturo Azcorra (2018).
Measurement Errors in \R. \emph{The R Journal}, 10(2), 549-557.
\doi{10.32614/RJ-2018-075}
}
\seealso{
\code{\link{datasets}} for a description of the datasets used in the
examples below.
}
\author{
Iñaki Ucar
}
