% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/estimate.R
\name{estimate}
\alias{estimate}
\title{Bayesian estimation of Structural Vector Autoregressions via Gibbs sampler}
\usage{
estimate(specification, S, thin = 1, show_progress = TRUE)
}
\arguments{
\item{specification}{an object generated using one of the \code{specify_bsvar*} functions
or an object generated using the function \code{estimate}.
The latter type of input facilitates the continuation of the MCMC sampling starting from the last draw of the previous run.}

\item{S}{a positive integer, the number of posterior draws to be generated}

\item{thin}{a positive integer, specifying the frequency of MCMC output thinning}

\item{show_progress}{a logical value, if \code{TRUE} the estimation progress bar is visible}
}
\value{
An object of class PosteriorBSVAR* containing the Bayesian estimation output and containing two elements:

\code{posterior} a list with a collection of \code{S} draws from the posterior distribution generated via Gibbs sampler containing many arrays and vectors whose selection depends on the model specification.
\code{last_draw} an object generated by one of the \code{specify_bsvar*} functions with the last draw of the current MCMC run as the starting value to be passed to the continuation of the MCMC estimation using \code{estimate()}.
}
\description{
Estimates homo- or heteroskedastic SVAR models for packages \pkg{bsvars}
and \pkg{bsvarSIGNs}. The packages apply the Gibbs sampler proposed by Waggoner & Zha (2003)
for the structural matrix \eqn{B} and the equation-by-equation sampler by Chan, Koop, & Yu (2024)
for the autoregressive slope parameters \eqn{A}. Additionally, the parameter matrices \eqn{A} and \eqn{B}
follow a Minnesota prior and generalised-normal prior distributions respectively with the matrix-specific
3-level equation-specific local-global hierarchical prior for the shrinkage parameters. A variety of models
for conditional variances are possible including versions of Stochastic Volatility and Markov-switching heteroskedasticity.
Non-normal specifications include finite and sparse normal mixture model for the structural shocks.
The estimation algorithms for particular models are scrutinised in 
Lütkepohl, Shang, Uzeda, & Woźniak (2024) and Woźniak & Droumaguet (2024)
and some other inferential and identification problems are considered in 
Lütkepohl & Woźniak (2020) and Song & Woźniak (2021).
Models from package \pkg{bsvars} implement identification via exclusion restrictions, 
heteroskedasticity and non-normality. Models from package \pkg{bsvarSIGNs} implement
identification via sign and narrative restrictions.
See section \bold{Details} and package \pkg{bsvarSIGNs} documentation for more information.
}
\details{
The homoskedastic SVAR model is given by the reduced form equation:
\deqn{Y = AX + E}
where \eqn{Y} is an \code{NxT} matrix of dependent variables, \eqn{X} is a \code{KxT} matrix of explanatory variables, 
\eqn{E} is an \code{NxT} matrix of reduced form error terms, and \eqn{A} is an \code{NxK} matrix of autoregressive slope coefficients and parameters on deterministic terms in \eqn{X}.

The structural equation is given by
\deqn{BE = U}
where \eqn{U} is an \code{NxT} matrix of structural form error terms, and
\eqn{B} is an \code{NxN} matrix of contemporaneous relationships.

The structural shocks, \code{U}, are temporally and contemporaneously independent and jointly normally distributed with zero mean and unit variances.

The various SVAR models estimated differ by the specification of structural shocks
variances. Their specification depends on the \code{specify_bsvar*} function used. The different models include:
\itemize{
  \item homoskedastic model with unit variances
  \item heteroskedastic model with stationary Markov switching in the variances
  \item heteroskedastic model with Stochastic Volatility process for variances
  \item non-normal model with a finite mixture of normal components and component-specific variances
  \item heteroskedastic model with sparse Markov switching in the variances where the number of heteroskedastic components is estimated
  \item non-normal model with a sparse mixture of normal components and component-specific variances where the number of heteroskedastic components is estimated
}
}
\examples{
# simple workflow
############################################################
# upload data
data(us_fiscal_lsuw)

# specify the model and set seed
specification  = specify_bsvar$new(us_fiscal_lsuw, p = 4)
set.seed(123)

# run the burn-in
burn_in        = estimate(specification, 5)

# estimate the model
posterior      = estimate(burn_in, 10, thin = 2)

# workflow with the pipe |>
############################################################
set.seed(123)
us_fiscal_lsuw |>
  specify_bsvar$new(p = 1) |>
  estimate(S = 5) |> 
  estimate(S = 10, thin = 2) -> posterior

}
\references{
Chan, J.C.C., Koop, G, and Yu, X. (2024) Large Order-Invariant Bayesian VARs with Stochastic Volatility. \emph{Journal of Business & Economic Statistics}, \bold{42}, \doi{10.1080/07350015.2023.2252039}.

Lütkepohl, H., Shang, F., Uzeda, L., and Woźniak, T. (2024) Partial Identification of Heteroskedastic Structural VARs: Theory and Bayesian Inference. \emph{University of Melbourne Working Paper}, 1--57, \doi{10.48550/arXiv.2404.11057}.

Lütkepohl, H., and Woźniak, T., (2020) Bayesian Inference for Structural Vector Autoregressions Identified by Markov-Switching Heteroskedasticity. \emph{Journal of Economic Dynamics and Control} \bold{113}, 103862, \doi{10.1016/j.jedc.2020.103862}.

Song, Y., and Woźniak, T. (2021) Markov Switching Heteroskedasticity in Time Series Analysis. In: \emph{Oxford Research Encyclopedia of Economics and Finance}. Oxford University Press, \doi{10.1093/acrefore/9780190625979.013.174}.

Waggoner, D.F., and Zha, T., (2003) A Gibbs sampler for structural vector autoregressions. \emph{Journal of Economic Dynamics and Control}, \bold{28}, 349--366, \doi{10.1016/S0165-1889(02)00168-9}.

Woźniak, T., and Droumaguet, M., (2024) Bayesian Assessment of Identifying Restrictions for Heteroskedastic Structural VARs.
}
\seealso{
\code{\link{specify_bsvar}}, \code{\link{specify_bsvar_msh}}, \code{\link{specify_bsvar_mix}}, \code{\link{specify_bsvar_sv}}, \code{\link{normalise_posterior}}
}
\author{
Tomasz Woźniak \email{wozniak.tom@pm.me}
}
