#' Coefficient matrices of endogenous variables
#'
#' Returns the estimated coefficient matrices of the lagged endogenous
#' variables of a VAR(p) model.
#' This is a modification of vars::Acoef() for the class "varshrinkest".
#'
#' Consider VAR(p) model:
#' \deqn{y_t = A_1 y_{t-1} + ... + A_p y_{t-p} + C d_t + e_t .}
#' The function returns the K-by-K matrices {A_1, ..., A_p} as a list object.
#'
#' @param x An object of class "varshrinkeset", generated by VARshrink().
#' @return A list object with K-by-K VAR coefficient matrices A_1, A_2, ..., A_p
#' @examples
#' data(Canada, package = "vars")
#' y <- diff(Canada)
#' estim <- VARshrink(y, p = 2, type = "const", method = "ridge")
#' Acoef_sh(estim)
#' @seealso \code{\link[vars]{Acoef}}
#' @export
Acoef_sh <- function (x) {

  if (!inherits(x, "varest")) {
    stop("\nPlease provide an object inheriting class 'varest'.\n")
  }
  K <- x$K
  p <- x$p
  A <- Bcoef_sh(x)[, 1:(K * p)]
  As <- list()
  start <- seq(1, p * K, K)
  end <- seq(K, p * K, K)
  for (i in 1:p) {
    As[[i]] <- matrix(A[, start[i]:end[i]], nrow = K, ncol = K)
    rownames(As[[i]]) <- rownames(A)
    colnames(As[[i]]) <- colnames(A[, start[i]:end[i]])
  }
  return(As)
}
