% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/update_mallows.R
\name{update_mallows}
\alias{update_mallows}
\alias{update_mallows.BayesMallows}
\alias{update_mallows.SMCMallows}
\title{Update a Bayesian Mallows model with new users}
\usage{
update_mallows(model, new_data, ...)

\method{update_mallows}{BayesMallows}(
  model,
  new_data,
  model_options = set_model_options(),
  smc_options = set_smc_options(),
  compute_options = set_compute_options(),
  priors = model$priors,
  ...
)

\method{update_mallows}{SMCMallows}(model, new_data, ...)
}
\arguments{
\item{model}{A model object of class "BayesMallows" returned from
\code{\link[=compute_mallows]{compute_mallows()}} or an object of class "SMCMallows" returned from this
function.}

\item{new_data}{An object of class "BayesMallowsData" returned from
\code{\link[=setup_rank_data]{setup_rank_data()}}. The object should contain the new data being provided.}

\item{...}{Optional arguments. Currently not used.}

\item{model_options}{An object of class "BayesMallowsModelOptions" returned
from \code{\link[=set_model_options]{set_model_options()}}.}

\item{smc_options}{An object of class "SMCOptions" returned from
\code{\link[=set_smc_options]{set_smc_options()}}.}

\item{compute_options}{An object of class "BayesMallowsComputeOptions"
returned from \code{\link[=set_compute_options]{set_compute_options()}}.}

\item{priors}{An object of class "BayesMallowsPriors" returned from
\code{\link[=set_priors]{set_priors()}}. Defaults to the priors used in \code{model}.}
}
\value{
An updated model, of class "SMCMallows".
}
\description{
Update a Bayesian Mallows model estimated using the Metropolis-Hastings
algorithm in \code{\link[=compute_mallows]{compute_mallows()}} using the sequential Monte Carlo algorithm
described in
\insertCite{steinSequentialInferenceMallows2023;textual}{BayesMallows}.
}
\examples{
set.seed(1)
# UPDATING A MALLOWS MODEL WITH NEW COMPLETE RANKINGS
# Assume we first only observe the first four rankings in the potato_visual
# dataset
data_first_batch <- potato_visual[1:4, ]

# We start by fitting a model using Metropolis-Hastings
mod_init <- compute_mallows(
  data = setup_rank_data(data_first_batch),
  compute_options = set_compute_options(nmc = 10000))

# Convergence seems good after no more than 2000 iterations
assess_convergence(mod_init)
mod_init$burnin <- 2000

# Next, assume we receive four more observations
data_second_batch <- potato_visual[5:8, ]

# We can now update the model using sequential Monte Carlo
mod_second <- update_mallows(
  model = mod_init, new_data = setup_rank_data(rankings = data_second_batch))

# This model now has a collection of particles approximating the posterior
# distribution after the first and second batch
# We can use all the posterior summary functions as we do for the model
# based on compute_mallows():
plot(mod_second)
plot(mod_second, parameter = "rho", items = 1:4)
compute_posterior_intervals(mod_second)

# Next, assume we receive the third and final batch of data. We can update
# the model again
data_third_batch <- potato_visual[9:12, ]
mod_final <- update_mallows(
  model = mod_second, new_data = setup_rank_data(rankings = data_third_batch))

# We can plot the same things as before
plot(mod_final)
compute_consensus(mod_final)

# UPDATING A MALLOWS MODEL WITH NEW OR UPDATED PARTIAL RANKINGS
# The sequential Monte Carlo algorithm works for data with missing ranks as
# well. This both includes the case where new users arrive with partial ranks,
# and when previously seen users arrive with more complete data than they had
# previously.
# We illustrate for top-k rankings of the first 10 users in potato_visual
potato_top_10 <- ifelse(potato_visual[1:10, ] > 10, NA_real_,
                        potato_visual[1:10, ])
potato_top_12 <- ifelse(potato_visual[1:10, ] > 12, NA_real_,
                        potato_visual[1:10, ])
potato_top_14 <- ifelse(potato_visual[1:10, ] > 14, NA_real_,
                        potato_visual[1:10, ])

# We need the rownames as user IDs
(user_ids <- rownames(potato_visual[1:10, ]))

# First, users provide top-10 rankings
mod_init <- compute_mallows(
  data = setup_rank_data(rankings = potato_top_10, user_ids = user_ids),
  compute_options = set_compute_options(nmc = 10000))

# Convergence seems fine. We set the burnin to 2000.
assess_convergence(mod_init)
mod_init$burnin <- 2000

# Next assume the users update their rankings, so we have top-12 instead.
mod1 <- update_mallows(
  model = mod_init,
  new_data = setup_rank_data(rankings = potato_top_12, user_ids = user_ids)
)

plot(mod1)

# Then, assume we get even more data, this time top-14 rankings:
mod2 <- update_mallows(
  model = mod1,
  new_data = setup_rank_data(rankings = potato_top_14, user_ids = user_ids)
)

plot(mod2)

# Finally, assume a set of new users arrive, who have complete rankings.
potato_new <- potato_visual[11:12, ]
# We need to update the user IDs, to show that these users are different
(user_ids <- rownames(potato_new))

mod_final <- update_mallows(
  model = mod2,
  new_data = setup_rank_data(rankings = potato_new, user_ids = user_ids)
)

plot(mod_final)
}
\seealso{
Other modeling: 
\code{\link{compute_mallows_mixtures}()},
\code{\link{compute_mallows}()}
}
\concept{modeling}
