/*
    FileManagerDelegate.h

    File manager delegate protocol declaration for the ProjectManager app.

    Copyright (C) 2006  Saso Kiselkov

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/

#import <Foundation/NSObject.h>

@class NSImage, NSString, NSError;

typedef enum {
  FileOpenCannotHandle = -1,
  FileOpenFailure,
  FileOpenSuccess
} FileOpenResult;

typedef enum {
  FileImportCannotHandle = -1,
  FileImportFailure,
  FileImportSuccess
} FileImportResult;

/**
 * This protocol declares what methods should be implemented by
 * a FileManager delegate object.
 */
@protocol FileManagerDelegate

/**
 * Should return YES if creating categories at the provided path is allowed,
 * and NO if it isn't.
 */
- (BOOL) canCreateCategoriesAtPath: (NSString *) aPath;

/**
 * Should return YES if creating or importing plain files at the
 * provided path is allowed, and NO if it isn't.
 */
- (BOOL) canCreatePlainFilesAtPath: (NSString *) aPath;

/**
 * Should return YES if creating links to files at the provided path
 * is allowed, and NO if it isn't.
 */
- (BOOL) canCreateLinksAtPath: (NSString *) aPath;

/**
 * Should return YES if creating virtual files at the provided path
 * is allowed, and NO if it isn't.
 */
- (BOOL) canCreateVirtualFilesAtPath: (NSString *) aPath;

/**
 * Should return YES if deleting a file at the provided path is allowed,
 * and NO if it isn't.
 */
- (BOOL) canDeletePath: (NSString *) aPath;

/**
 * Asks the file manager's delegate to open the file at the specified
 * project path.
 *
 * @return If the delegate can open the file, it should return either
 * FileOpenSuccess or FileOpenFailure, depending on whether opening
 * the file succeeded or not. If it can't open the file, or would
 * rather like the FileManager try to do that, it should return
 * FileOpenCannotHandle instead.
 */
- (FileOpenResult) openFile: (NSString *) aPath;

/**
 * Should return a path to the physical location of the specified path
 * on disk.
 *
 * @param aPath The virtual path for which to return the physical path.
 * @param isCategory Specifies whether the object identified by `aPath'
 *      is a category or a file.
 *
 * @return A path to the physical location of the file.
 */
- (NSString *) pathToFile: (NSString *) aPath isCategory: (BOOL) isCategory;

/**
 * Should return an icon for the specified path, if the delegate wishes
 * to assign a special icon to that file. If the default icon for the
 * file is to be used, `nil' should be returned instead.
 */
- (NSImage *) iconForPath: (NSString *) aPath;

/**
 * Should return an array of file extensions for files which are
 * permitted in the specified category. If any files are allowed, `nil'
 * should be returned instead.
 */
- (NSArray *) permissibleFileExtensionsInCategory: (NSString *) aCategory;

/**
 * Asks the delegate to handle a file import into the project. The delegate
 * can use this method to override the default importing mechanisms, which
 * would copy or link the source file and instead invoke some more complex
 * mechanisms.
 *
 * @param aFile The file in the filesystem which to import.
 * @param aCategory The category where to import the file.
 * @param error A pointer which should be set to an NSError object
 *      describing the error in case the operation failed.
 *
 * @return The result of the import. The possibilities mean:
 * - FileImportCannotHandle --  tells the file manager to execute
 *      the standard importing procedure.
 * - FileImportFailure -- indicates that the import has failed.
 * - FileImportSuccess -- indicates that the import has succeeded.
 */
- (FileImportResult) importFile: (NSString *) aFile
                   intoCategory: (NSString *) aCategory
                          error: (NSError **) error;

/**
 * Should return a path to a directory containing file templates
 * for files that are created in category `aCategory'. In case there
 * are no templates available for that category, `nil' should be returned
 * instead.
 *
 * The templates directories are directories which follow these rules:
 *
 * - they may contain subdirectories nested to any depth, but these
 *   subdirectories cannot contain any path extension, to allow the
 *   templates machinery to clearly distinguish between sub-directories
 *   and file packages (e.g. ".gorm" files).
 * - an other files are treated as template files, except for
 * - a file named like any respective template file with the special
 *   extension ".pmdesc" added - this file should contain a short textual
 *   description of the specific file.
 *
 * An example templates directory structure:
 *
 * +-- <dir> "My_Files_Template.templates"
 *      |
 *      +-- <dir> "Simple Classes"
 *      |    |
 *      |    +-- <file> "NSObject Subclass.m"
 *      |    |
 *      |    +-- <file> "NSObject Subclass.m.pmdesc"
 *      |
 *      |
 *      +-- <dir> "NSView Subclasses"
 *      |    |
 *      |    +-- <file> "NSControl Subclass.m"
 *      |    |
 *      |    +-- <file> "NSControl Subclass.m.pmdesc"
 *      |
 *      +-- <dir> "Interface Files"
 *           |
 *           +-- <dir> "Main Menu.gorm"
 *           |    |
 *           |    +-- <file> "objects.gorm"
 *           |    |
 *           |    +-- <file> "data.info"
 *           |    |
 *           |    +-- <file> "data.classes"
 *           |
 *           +-- ...
 */
- (NSString *) pathToFileTemplatesDirectoryForCategory: (NSString *) aCategory;

/**
 * Should return a list of files which are to be additionally added to the
 * project into specific categories, when the user chose to add the
 * template file `aFile' from `templatesDir' into the project category
 * `targetCategory'.
 *
 * The associated files are assigned the same name as the user chose for
 * the created file, but keep their file extensions.
 *
 * @return A dictionary describing a list of files like this:
 * {
 *   "<file-path>" = "<category-where-to-import-it>";
 *   ...
 * }
 */
- (NSDictionary *) filesAssociatedWithTemplateFile: (NSString *) aFile
                            fromTemplatesDirectory: (NSString *) templatesDir
                                       forCategory: (NSString *) targetCategory;

@end
