// Copyright (c) 2014-2017 Josh Blum
// SPDX-License-Identifier: BSL-1.0

#include <Pothos/Framework.hpp>
#include <algorithm> //min/max

/***********************************************************************
 * |PothosDoc Stream To Packet
 *
 * The stream to packet block is a simple translator between
 * a byte stream input and a message-based packetized output.
 * The block accepts a byte stream with labels on input port 0,
 * and converts the stream into Pothos::Packet message objects.
 * The packet message objects are then produced on output port 0.
 *
 * If the input port 0 has an incoming message,
 * it will be forwarded directly to output port 0.
 *
 * This is zero-copy block implementation.
 * The output packet object holds a reference to the input stream buffer,
 * without incurring a copy of the buffer.
 *
 * <h2>MTU usage</h2>
 * The MTU can be used to control the maximum size packet payload.
 * When specified, the packet payloads cannot exceed the MTU size.
 * However, packet payloads may be smaller than the specified MTU
 * if the available buffer was smaller.
 * When the MTU is unspecified, the entire available buffer is used.
 *
 * <h2>Label support</h2>
 *
 * Labels can be used to indicate frame boundaries,
 * and to discard stream data located outside of the boundaries.
 *
 * <b>Default operation:</b>
 * In the default operation mode, no frame IDs are specified.
 * The stream to packet block accepts and forwards the input buffer.
 *
 * <b>Start-frame operation:</b>
 * In the start-frame operation mode, no end of frame ID is specified.
 * The stream to packet block drops all input data
 * until a start of frame label is encountered.
 * Unlike the other modes which treat MTU as a maximum length,
 * this mode produces an exact MTU length payload for every start of frame.
 * If the start frame label contains an element count length,
 * then the MTU is overridden and the specified length is used.
 *
 * <b>Full-frame operation:</b>
 * In full-frame operation mode, both frame IDs are specified.
 * The stream to packet block drops all input data
 * until a start of frame label is encountered.
 * After that, multiple packet payloads are produced
 * until an end of frame label is encountered.
 *
 * |category /Packet
 * |category /Convert
 * |keywords packet message datagram
 * |alias /blocks/label_deframer
 *
 * |param mtu[MTU] The maximum size of the payload in an output packet.
 * An MTU of 0 bytes means unconstrained payload size;
 * packet payloads will accept the entire input buffer.
 * |default 0
 * |units bytes
 * |preview valid
 *
 * |param frameStartId[Frame Start ID] The label ID to mark the first element from each payload.
 * An empty string (default) means that start of frame labels are not produced.
 * |default ""
 * |widget StringEntry()
 * |preview valid
 *
 * |param frameEndId[Frame End ID] The label ID to mark the last element from each payload.
 * An empty string (default) means that end of frame labels are not produced.
 * |default ""
 * |widget StringEntry()
 * |preview valid
 *
 * |factory /blocks/stream_to_packet()
 * |setter setMTU(mtu)
 * |setter setFrameStartId(frameStartId)
 * |setter setFrameEndId(frameEndId)
 **********************************************************************/
class StreamToPacket : public Pothos::Block
{
public:
    StreamToPacket(void):
        _mtu(0),
        _inFrame(false),
        _startFrameMode(false),
        _fullFrameMode(false)
    {
        this->setupInput(0);
        this->setupOutput(0);
        this->registerCall(this, POTHOS_FCN_TUPLE(StreamToPacket, setMTU));
        this->registerCall(this, POTHOS_FCN_TUPLE(StreamToPacket, getMTU));
        this->registerCall(this, POTHOS_FCN_TUPLE(StreamToPacket, setFrameStartId));
        this->registerCall(this, POTHOS_FCN_TUPLE(StreamToPacket, getFrameStartId));
        this->registerCall(this, POTHOS_FCN_TUPLE(StreamToPacket, setFrameEndId));
        this->registerCall(this, POTHOS_FCN_TUPLE(StreamToPacket, getFrameEndId));
    }

    static Block *make(void)
    {
        return new StreamToPacket();
    }

    void setMTU(const size_t mtu)
    {
        _mtu = mtu;
    }

    size_t getMTU(void) const
    {
        return _mtu;
    }

    void setFrameStartId(std::string id)
    {
        _frameStartId = id;
        this->updateModes();
    }

    std::string getFrameStartId(void) const
    {
        return _frameStartId;
    }

    void setFrameEndId(std::string id)
    {
        _frameEndId = id;
        this->updateModes();
    }

    std::string getFrameEndId(void) const
    {
        return _frameEndId;
    }

    void activate(void)
    {
        _inFrame = false; //reset state
    }

    void work(void)
    {
        auto inputPort = this->input(0);
        auto outputPort = this->output(0);

        //forward messages
        while (inputPort->hasMessage())
        {
            auto msg = inputPort->popMessage();
            outputPort->postMessage(std::move(msg));
        }

        //is there any input buffer available?
        if (inputPort->elements() == 0) return;

        //start frame mode has its own work implementation
        if (_startFrameMode) return this->startFrameModeWork();

        //drop until start of frame label
        if (_fullFrameMode and not _inFrame)
        {
            for (const auto &label : inputPort->labels())
            {
                //end of input buffer labels, exit loop
                if (label.index >= inputPort->elements()) break;

                //ignore labels that are not start of frame
                if (label.id != _frameStartId) continue;

                //in position 0, set in frame, done loop
                if (label.index == 0)
                {
                    _inFrame = true;
                    break;
                }

                //otherwise consume up to but not including
                //done work, will be in-frame for next work
                else
                {
                    inputPort->consume(label.index);
                    _inFrame = true;
                    return;
                }
            }

            //start of frame not found, consume everything, exit this work
            if (not _inFrame)
            {
                inputPort->consume(inputPort->elements());
                return;
            }
        }

        //grab the input buffer
        Pothos::Packet packet;
        packet.payload = inputPort->takeBuffer();
        if (_mtu != 0)
        {
            const auto elemSize = packet.payload.dtype.size();
            const auto mtuElems = (_mtu/elemSize)*elemSize;
            packet.payload.length = std::min(mtuElems, packet.payload.length);
        }

        //grab the input labels
        for (const auto &label : inputPort->labels())
        {
            const auto pktLabel = label.toAdjusted(
                1, packet.payload.dtype.size()); //bytes to elements
            if (pktLabel.index >= packet.payload.elements()) break;
            packet.labels.push_back(std::move(pktLabel));

            //end of frame found, truncate payload and leave loop
            if (_fullFrameMode and label.id == _frameEndId)
            {
                packet.payload.length = label.index+label.width;
                _inFrame = false;
                break;
            }
        }

        //consume the input and produce the packet
        inputPort->consume(packet.payload.length);
        outputPort->postMessage(std::move(packet));
    }

    /*******************************************************************
     * start frame operation mode work:
     * The implementation was sufficiently different to separate.
     ******************************************************************/
    void startFrameModeWork(void)
    {
        auto inputPort = this->input(0);
        auto outputPort = this->output(0);

        //get input buffer
        auto inBuff = inputPort->buffer();
        if (inBuff.length == 0) return;

        bool frameFound = false;
        Pothos::Packet packet;
        packet.payload = inBuff;
        size_t outputLength = _mtu;

        auto inLen = inBuff.length;
        for (auto &label : inputPort->labels())
        {
            // Skip any label that doesn't yet appear in the data buffer
            if (label.index >= inLen) continue;

            // Skip any label that isn't a frame start label
            if (label.id != _frameStartId) continue;

            //use the label's length when specified
            if (label.data.canConvert(typeid(size_t)))
            {
                outputLength = label.data.convert<size_t>();
                outputLength *= label.width; //expand for width
                outputLength *= inBuff.dtype.size(); //convert to bytes
            }

            // Skip all of data before the start of packet if this is the first time we see the label
            if (label.index != 0)
            {
                inputPort->consume(label.index);
                inputPort->setReserve(outputLength);
                return;
            }

            // This case happens when the start of frame is naturally aligned with the begining of a buffer, but we didn't get enough data
            // In that case we wait
            if (inLen < outputLength)
            {
                inputPort->setReserve(outputLength);
                return;
            }

            //found! start of frame on index 0
            frameFound = true;
            break;
        }

        // Skip all of the data in case we didn't see any frame labels
        if (not frameFound)
        {
            inputPort->consume(inLen);
            return;
        }

        //load non-frame start labels into the packet
        size_t nextFrameIndex = 0;
        for (auto &label : inputPort->labels())
        {
            if (label.index >= outputLength) continue;
            if (label.id == _frameStartId)
            {
                if (nextFrameIndex == 0)
                {
                    nextFrameIndex = label.index;
                }
                continue;
            }
            packet.labels.push_back(label);
        }

        //produce the output packet
        packet.payload.length = outputLength;
        outputPort->postMessage(packet);
        inputPort->setReserve(0); //clear reserve for next packet

        //consume the input, stopping at the next frame label (in the case of overlap)
        inputPort->consume((nextFrameIndex != 0)?nextFrameIndex:outputLength);
    }

    void propagateLabels(const Pothos::InputPort *)
    {
        //labels are not propagated
        return;
    }

    //the circular buffer allows stream to packet to forwards maximum sized payloads of available input buffer
    //its also useful for efficiency purposes when running in the start of frame only mode
    Pothos::BufferManager::Sptr getInputBufferManager(const std::string &, const std::string &)
    {
        return Pothos::BufferManager::make("circular");
    }

private:

    void updateModes(void)
    {
        _startFrameMode = not _frameStartId.empty() and _frameEndId.empty();
        _fullFrameMode = not _frameStartId.empty() and not _frameEndId.empty();
    }

    size_t _mtu;
    std::string _frameStartId;
    std::string _frameEndId;
    bool _inFrame;
    bool _startFrameMode;
    bool _fullFrameMode;
};

static Pothos::BlockRegistry registerStreamToPacket(
    "/blocks/stream_to_packet", &StreamToPacket::make);

//backwards compatible alias
static Pothos::BlockRegistry registerLabelDeframer(
    "/blocks/label_deframer", &StreamToPacket::make);
