/*
**  PlopWindowController.m
**
**  Copyright (c) 2002
**
**  Author: Ludovic Marcotte <ludovic@Sophos.ca>
**
**  This program is free software; you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation; either version 2 of the License, or
**  (at your option) any later version.
**
**  This program is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with this program; if not, write to the Free Software
**  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#import "PlopWindowController.h"

#import "Channel.h"
#import "Constants.h"
#import "NSStringExtensions.h"
#import "Plop.h"
#import "PlopView.h"
#import "PlopWindow.h"

static id lastPlopWindowOnTop = nil;

@implementation PlopWindowController

- (id) initWithWindowNibName: (NSString *) theNibName
{
  PlopWindow *plopWindow;
  
  plopWindow = [[PlopWindow alloc] initWithContentRect:NSMakeRect(10,10,DEFAULT_PLOP_WIDTH,DEFAULT_PLOP_HEIGHT)
				   styleMask: NSBorderlessWindowMask
                                   backing: NSBackingStoreBuffered
                                   defer: NO];

  self = [super initWithWindow: plopWindow];
  RELEASE(plopWindow);
 
  [plopWindow layoutWindow];
  [plopWindow setDelegate: self];

  // We link our outlets
  textView = [plopWindow textView];


  // We set our text view color
  [textView setBackgroundColor: [NSColor colorWithDeviceRed: 0.90
					 green: 0.90
					 blue: 0.90
					 alpha: 1.0]];
  
  // We set the title to an empty string
  [[self window] setTitle: @""];
 
  return self;
}


//
//
//
- (void) dealloc
{
  NSLog(@"PlopWindowController: -dealloc");

  lastPlopWindowOnTop = nil;
  
  [channel setPlopWindowController: nil];
  RELEASE(channel);
  
  [super dealloc];
}



//
// delegate methods
//
- (BOOL) textView: (NSTextView *) textView
    clickedOnLink: (id) link 
	  atIndex: (unsigned) charIndex
{
  return [self openURL: link];
}


//
//
//
- (void) windowDidBecomeKey: (NSNotification *) aNotification
{
  lastPlopWindowOnTop = [self window];
  //[[PlopInfoWindowController singleInstance] setPlop: [self plop]];
}


//
//
//
- (void) windowDidLoad
{
  lastPlopWindowOnTop = [self window];
  
  [super windowDidLoad];
}


//
//
//
- (void) windowDidMove: (NSNotification *) aNotification
{
  [self _updateWindowFrame];
}


//
//
//
- (void) windowDidResize: (NSNotification *) aNotification
{
  [self _updateWindowFrame];
}


//
//
//
- (void) windowWillClose: (NSNotification *) theNotification
{
  AUTORELEASE(self);
}


//
// access / mutation
//
+ (id) lastPlopWindowOnTop
{
  return lastPlopWindowOnTop;
}


//
//
//
- (Channel *) channel
{
  return channel;
}


//
//
//
- (void) setChannel: (Channel *) theChannel
{
  if ( theChannel )
    {
      RETAIN(theChannel);
      RELEASE(channel);
      channel = theChannel;

      [channel setPlopWindowController: self];
      
      // We set our count variable and our icon
      [[self plopView] setCount: [channel itemCount]];
      [[self plopView] setIcon: [channel icon]];
      
      // We now update the text view with the content of the channel
      [self update];

      // We finally set the window title and frame
      if ( [channel title] )
        {
          [[self window] setTitle: [channel title]];
        }
      [[self window] setFrame: [[channel plop] frame]
		     display: NO];
    }
  else
    {
      RELEASE(channel);
      channel = nil;
    }
}


//
//
//
- (NSTextView *) textView
{
  return textView;
}


//
//
//
- (PlopView *) plopView
{
  return [(PlopWindow *)[self window] plopView];
}



//
// Other methods
//
- (BOOL) openURL: (NSURL *) theURL
{
  NSLog(@"Opening %@...", [theURL description]);

  if ( [[NSUserDefaults standardUserDefaults] integerForKey: @"BROWSER_SOURCE"] == USE_DEFAULT_BROWSER )
    {
      return [[NSWorkspace sharedWorkspace] openURL: theURL];
    }
  else
    {
      NSString *handler;

      handler = [[NSUserDefaults standardUserDefaults] objectForKey: @"BROWSER_PATH"];

      if ( !handler || [[handler stringByTrimmingWhiteSpaces] length] == 0 )
	{
	  NSBeep();
	  return NO;
	}
      
      [NSTask launchedTaskWithLaunchPath: @"/bin/sh"
	      arguments: [NSArray arrayWithObjects: @"-c",
				  [NSString stringWithFormat: handler, [theURL description]],
				  nil] ];

    }
  
  return YES;
}
//
//
//
- (void) update
{
  NSMutableAttributedString *aMutableAttributedString;
  int i;
  
  if ( ![self channel] )
    {
      return;
    }
  
  aMutableAttributedString = [[NSMutableAttributedString alloc] init];
  
  for (i = 0; i < [[self channel] count]; i++)
    {
      NSDictionary *linkAttributes;
      Element *aElement;
      
      aElement = [[self channel] elementAtIndex: i];
      
      if ( [aElement type] == ITEM && [aElement link] )
	{
	  NSAttributedString *aAttributedString;
	  linkAttributes= [NSDictionary dictionaryWithObjectsAndKeys: [aElement link], NSLinkAttributeName,
#ifdef MACOSX
					[NSNumber numberWithInt: NSNoUnderlineStyle],
#else
					[NSNumber numberWithInt: GSNoUnderlineStyle],	
#endif
					NSUnderlineStyleAttributeName,
					[NSColor blackColor], NSForegroundColorAttributeName,
					NULL];
	  aAttributedString = [[NSAttributedString alloc] initWithString: [[aElement title] 
									    stringByAppendingString: @"\n"]
							  attributes: linkAttributes];
	  
	  [aMutableAttributedString appendAttributedString: aAttributedString];
	}
    }
  
  [[textView textStorage] setAttributedString: aMutableAttributedString];
  RELEASE(aMutableAttributedString);

  // We set the count of plopview and ask to refresh it
  [[self plopView] setCount: [channel itemCount]];
  [[self plopView] setNeedsDisplay: YES];
}


@end


//
// private methods
//
@implementation PlopWindowController (Private)

- (void) _updateWindowFrame
{
  if ( [[self channel] plop] )
    {
      NSRect theFrame;

      theFrame = [[self window] frame];
      [[[self channel] plop] setFrame: theFrame];
    }
}


@end
