//
//  MODDatabase.m
//  MongoObjCDriver
//
//  Created by Jérôme Lebel on 03/09/2011.
//

#import "MOD_internal.h"

@interface MODDatabase ()
@property (nonatomic, readwrite, strong) MODClient *client;
@property (nonatomic, readwrite, copy) NSString *name;

@end

@implementation MODDatabase

@synthesize client = _client, name = _name, mongocDatabase = _mongocDatabase, readPreferences = _readPreferences;

- (instancetype)initWithClient:(MODClient *)client name:(NSString *)name
{
    if (self = [self init]) {
        self.client = client;
        self.name = name;
        self.mongocDatabase = mongoc_client_get_database(client.mongocClient, self.name.UTF8String);
    }
    return self;
}

- (void)dealloc
{
    self.client = nil;
    if (self.mongocDatabase) {
        mongoc_database_destroy(self.mongocDatabase);
        self.mongocDatabase = nil;
    }
    [_systemIndexesCollection release];
    [super dealloc];
}

- (void)mongoQueryDidFinish:(MODQuery *)mongoQuery withBsonError:(bson_error_t)error callbackBlock:(void (^)(void))callbackBlock
{
    [self.client mongoQueryDidFinish:mongoQuery withBsonError:error callbackBlock:callbackBlock];
}

- (void)mongoQueryDidFinish:(MODQuery *)mongoQuery withError:(NSError *)error callbackBlock:(void (^)(void))callbackBlock
{
    [self.client mongoQueryDidFinish:mongoQuery withError:error callbackBlock:callbackBlock];
}

- (MODQuery *)statsWithReadPreferences:(MODReadPreferences *)readPreferences callback:(void (^)(MODSortedMutableDictionary *databaseStats, MODQuery *mongoQuery))callback;
{
    MODQuery *query;
    
    query = [self.client addQueryInQueue:^(MODQuery *mongoQuery){
        MODSortedMutableDictionary *stats = nil;
        bson_error_t error = BSON_NO_ERROR;
        
        if (!mongoQuery.isCanceled) {
            bson_t cmd = BSON_INITIALIZER;
            bson_t output;
            
            bson_init (&cmd);
            BSON_APPEND_INT32(&cmd, "dbstats", 1);
            if (mongoc_database_command_simple(self.mongocDatabase, &cmd, readPreferences?readPreferences.mongocReadPreferences:NULL, &output, &error)) {
                stats = [[self.client class] objectFromBson:&output];
            }
            bson_destroy(&cmd);
            bson_destroy(&output);
        }
        [self mongoQueryDidFinish:mongoQuery withBsonError:error callbackBlock:^(void) {
            if (!mongoQuery.isCanceled && callback) {
                callback(stats, mongoQuery);
            }
        }];
    } owner:self name:@"databasestats" parameters:nil];
    return query;
}

- (MODQuery *)collectionNamesWithCallback:(void (^)(NSArray *collectionList, MODQuery *mongoQuery))callback;
{
    MODQuery *query;
    
    query = [self.client addQueryInQueue:^(MODQuery *mongoQuery){
        NSMutableArray *collections = nil;
        bson_error_t error = BSON_NO_ERROR;
        
        if (!mongoQuery.isCanceled) {
            char **cStringCollections;
            
            cStringCollections = mongoc_database_get_collection_names(self.mongocDatabase, &error);
            if (cStringCollections) {
                char **cursor = cStringCollections;
                
                collections = [[NSMutableArray alloc] init];
                while (*cursor != NULL) {
                    [collections addObject:[NSString stringWithUTF8String:*cursor]];
                    bson_free(*cursor);
                    cursor++;
                }
                bson_free(cStringCollections);
            }
        }
        [self mongoQueryDidFinish:mongoQuery withBsonError:error callbackBlock:^(void) {
            if (!mongoQuery.isCanceled && callback) {
                callback(collections, mongoQuery);
            }
        }];
        [collections release];
    } owner:self name:@"collectionnames" parameters:nil];
    return query;
}

- (MODCollection *)collectionForName:(NSString *)name
{
    MODCollection *result;
    
    result = [[[MODCollection alloc] initWithName:name database:self] autorelease];
    result.readPreferences = self.readPreferences;
    return result;
}

- (MODQuery *)createCollectionWithName:(NSString *)collectionName callback:(void (^)(MODQuery *mongoQuery))callback
{
    MODQuery *query = nil;
    
    query = [self.client addQueryInQueue:^(MODQuery *mongoQuery){
        bson_error_t error = BSON_NO_ERROR;
        
        if (!mongoQuery.isCanceled) {
            mongoc_database_create_collection(self.mongocDatabase, collectionName.UTF8String, NULL, &error);
        }
        [self mongoQueryDidFinish:mongoQuery withBsonError:error callbackBlock:^(void) {
            if (!mongoQuery.isCanceled && callback) {
                callback(mongoQuery);
            }
        }];
    } owner:self name:@"createcollection" parameters:@{ @"collectionname": collectionName }];
    return query;
}

//- (MODQuery *)createCappedCollectionWithName:(NSString *)collectionName capSize:(int64_t)capSize callback:(void (^)(MODQuery *mongoQuery))callback
//{
//    MODQuery *query;
//    
//    query = [self.client addQueryInQueue:^(MODQuery *mongoQuery){
//        if (!self.client.isMaster) {
//            mongoQuery.error = [MODClient errorWithErrorDomain:MODMongoErrorDomain code:MONGO_CONN_NOT_MASTER descriptionDetails:@"Collection add forbidden on a slave"];
//        } else if (!mongoQuery.isCanceled && [self.client authenticateSynchronouslyWithDatabaseName:_databaseName userName:_userName password:_password mongoQuery:mongoQuery]) {
//            mongo_cmd_create_capped_collection(self.client.mongo, [_databaseName UTF8String], [collectionName UTF8String], capSize);
//        }
//        [self mongoQueryDidFinish:mongoQuery withCallbackBlock:^(void) {
//            if (callback) {
//                callback(mongoQuery);
//            }
//        }];
//    }];
//    [query.mutableParameters setObject:@"createcappedcollection" forKey:@"command"];
//    [query.mutableParameters setObject:collectionName forKey:@"collectionname"];
//    [query.mutableParameters setObject:[NSNumber numberWithLongLong:capSize] forKey:@"capsize"];
//    return query;
//}

- (MODQuery *)dropWithCallback:(void (^)(MODQuery *mongoQuery))callback
{
    MODQuery *query;
    
    query = [self.client addQueryInQueue:^(MODQuery *mongoQuery) {
        bson_error_t error = BSON_NO_ERROR;
        
        if (!mongoQuery.isCanceled) {
            mongoc_database_drop(self.mongocDatabase, &error);
        }
        [self mongoQueryDidFinish:mongoQuery withBsonError:error callbackBlock:^(void) {
            if (!mongoQuery.isCanceled && callback) {
                callback(mongoQuery);
            }
        }];
    } owner:self name:@"dropdatabase" parameters:@{ @"name": self.name }];
    return query;
}

- (mongoc_client_t *)mongocClient
{
    return self.client.mongocClient;
}

- (MODCollection *)systemIndexesCollection
{
    if (!_systemIndexesCollection) {
        _systemIndexesCollection = [[self collectionForName:@"system.indexes"] retain];
    }
    return _systemIndexesCollection;
}

- (mongoc_read_prefs_t *)mongocReadPreferences
{
    return self.readPreferences.mongocReadPreferences;
}

- (void)setReadPreferences:(MODReadPreferences *)readPreferences
{
    [_readPreferences release];
    _readPreferences = [readPreferences retain];
    if (self.mongocDatabase) {
        mongoc_database_set_read_prefs(self.mongocDatabase, self.mongocReadPreferences);
    }
    
}

@end
