#include <config.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <pthread.h>
#include <errno.h>
#include <sys/sysctl.h>

#include "global.h"
#include "mpegconsts.h"
#include "yuv4mpeg.h"

/**/
#include <assert.h>

//#define DEMO
#ifdef DEMO
#warning "Compiling in DEMO mode..."
#include "imgtable.h"
unsigned int demo_pic_count;
#else
#warning "Compiling in RELEASE mode..."
#endif

#include "mpeg2.h"
void *mpeg2dec;
#define RAWBUFSIZE 4096
unsigned char rawbuf[RAWBUFSIZE];
int pictSize;
int gotFirstPict;

struct _pict_info
{
	int type; //1 = i, 2 = p, 3 = b
	int tempref;
	int prog;
	int tff;
	int repeat;
	//double quantiser;
	int64_t bitcount;
	int pictSize;
	char seqHead;
};
typedef struct _pict_info pict_info_s;

static pict_info_s *pinfos = NULL;
/**/

static pthread_mutex_t frame_buffer_lock;

static pthread_cond_t new_chunk_req = PTHREAD_COND_INITIALIZER;
static pthread_cond_t new_chunk_ack = PTHREAD_COND_INITIALIZER;
static pthread_t      worker_thread;

static volatile int frames_read = 0;
static int last_frame = -1;

static void read_chunk(void)
{
	int n, j, i, gotFrame, y, state;
	const mpeg2_info_t *info = mpeg2_info(mpeg2dec);
	
	for(j=0;j<READ_CHUNK_SIZE;++j)
	{
		if( ctl_parallel_read ) pthread_mutex_unlock( &frame_buffer_lock );
		
		n = frames_read % frame_buffer_size;
		gotFrame = 0;
		
		char seqHead = 0;
		while (!gotFrame)
		{
			state = mpeg2_parse(mpeg2dec);
			switch(state)
			{
				case STATE_SEQUENCE:
				case STATE_SEQUENCE_REPEATED:
					seqHead = 1;
					break;
			
				case STATE_SLICE:
				case STATE_END:
					if (info->current_fbuf)
					{
						int w = info->sequence->width, h = info->sequence->height;
						
						for (y=0;y<h;y++) memcpy(frame_buffers[n][0]+y*opt_phy_width,info->current_fbuf->buf[0]+y*w,w);
						h >>= 1; w >>= 1;
						for (y=0;y<h;y++) memcpy(frame_buffers[n][1]+y*opt_phy_chrom_width,info->current_fbuf->buf[1]+y*w,w);
						for (y=0;y<h;y++) memcpy(frame_buffers[n][2]+y*opt_phy_chrom_width,info->current_fbuf->buf[2]+y*w,w);

#ifdef DEMO
						if (demo_pic_count % (25*60) < (25*5)) // 5 seconds each minute
						{
							const unsigned char *img = OVERLAY, *opa = ALPHA;
							unsigned char	*dstY = frame_buffers[n][0];

							int posx = w - (IMG_WIDTH >> 1), // w & h are currently chroma size.
								posy = h - (IMG_HEIGHT >> 1),
								x;
							
							dstY += posy * opt_phy_width;
							
							for (y = 0; y < IMG_HEIGHT; y++)
							{
								for (x = posx; x < (posx + IMG_WIDTH); x++)
								{
									dstY[x] = ((*img * *opa) + (dstY[x] * (0xFF - *opa))) >> 8;
									img++; opa++;
								}
								
								dstY += opt_phy_width;
							}
						}
						demo_pic_count++;
#endif
						gotFrame = 1;
						
						int pos = mpeg2_getpos(mpeg2dec);
						pictSize -= pos;
						//fprintf(stderr, "pos: %i\n", pos);
						//fprintf(stderr, "s: %i\n", pictSize);
						//assert(pictSize > 0);
						if (pictSize <= 0)
						{
							fprintf(stderr, "ill. pict. size: %i\n", pictSize);
							pictSize = 1;
						}
						
						/* get frame info */
						pinfos[n].tempref = info->current_picture->temporal_reference;
						pinfos[n].type = info->current_picture->flags & PIC_MASK_CODING_TYPE;
						pinfos[n].tff = (info->current_picture->flags & PIC_FLAG_TOP_FIELD_FIRST) ? 1 : 0;
						pinfos[n].repeat = (info->current_picture->nb_fields == 3) ? 1 : 0;
						pinfos[n].prog = (info->current_picture->flags & PIC_FLAG_PROGRESSIVE_FRAME) ? 1 : 0;
						//pinfos[n].quantiser = info->current_picture->quantiser;
						pinfos[n].bitcount = ibitcount;
						pinfos[n].pictSize = pictSize;
						pinfos[n].seqHead = seqHead;
						if (seqHead) seqHead = 0;
						if (gotFirstPict == 0)
						{
							gotFirstPict = 1;
							pinfos[n].seqHead = 1;
						}
						
						pictSize = pos; 
					}
					break;
	
				case STATE_BUFFER:
					i = read(istrm_fd,rawbuf,RAWBUFSIZE);
					if (i<=0) goto EOF_MARK;
					ibitcount += i << 3; pictSize += i;
					mpeg2_buffer(mpeg2dec,rawbuf,rawbuf+i);
					break;
			}
		}

		if( ctl_parallel_read ) pthread_mutex_lock( &frame_buffer_lock );
		++frames_read;
		if( ctl_parallel_read ) pthread_cond_broadcast( &new_chunk_ack );
	}

	return;
	
EOF_MARK:
	
	if( ctl_parallel_read ) pthread_mutex_lock( &frame_buffer_lock );
	last_frame = frames_read-1;
	istrm_nframes = frames_read;
	if( ctl_parallel_read ) pthread_cond_broadcast( &new_chunk_ack );
}

static void *read_chunks_worker(void *_dummy)
{
    pthread_mutex_lock( &frame_buffer_lock );
    read_chunk();
	for(;;)
	{
		pthread_cond_wait( &new_chunk_req, &frame_buffer_lock );
		if( frames_read < istrm_nframes ) read_chunk();
	}
	return NULL;
}


static void start_worker(void)
{
	pthread_attr_t *pattr = NULL;

#ifdef HAVE_PTHREADSTACKSIZE
#define MINSTACKSIZE 200000
	pthread_attr_t attr;
	size_t stacksize;
	
	pthread_attr_init(&attr);
	pthread_attr_getstacksize(&attr, &stacksize);
	
	if (stacksize < MINSTACKSIZE) pthread_attr_setstacksize(&attr, MINSTACKSIZE);
	pattr = &attr;
#endif

	if( pthread_create( &worker_thread, pattr, read_chunks_worker, NULL ) != 0 )
	{
		mjpeg_error_exit1( "worker thread creation failed: %s", strerror(errno) );
	}

}

   
static void read_chunk_seq( int num_frame )
{
    while(frames_read - num_frame < READ_CHUNK_SIZE && frames_read < istrm_nframes ) 
        read_chunk();
}


static void read_chunk_par( int num_frame)
{
	pthread_mutex_lock( &frame_buffer_lock);
	for(;;)
	{
		if( frames_read - num_frame < READ_CHUNK_SIZE &&  frames_read < istrm_nframes )
		{
			pthread_cond_broadcast( &new_chunk_req );
		}
		if( frames_read > num_frame  || frames_read >= istrm_nframes )
		{
			pthread_mutex_unlock( &frame_buffer_lock );
			return;
		}
		pthread_cond_wait( &new_chunk_ack, &frame_buffer_lock );
	}
}

static void load_frame( int num_frame )
{
	
	if(last_frame>=0 && num_frame>last_frame && num_frame<istrm_nframes)
	{
		mjpeg_error("Internal:readframe: internal error reading beyond end of frames");
		abort();
	}
	
	if( frames_read == 0)
	{
		pthread_mutexattr_t *p_attr = NULL;		
		pthread_mutex_init( &frame_buffer_lock, p_attr );
		
		pinfos = malloc(frame_buffer_size*sizeof(pict_info_s));

		if( ctl_parallel_read )
        {
			start_worker();
            read_chunk_par( num_frame);
        }
        else read_chunk_seq( num_frame);
	}

   if( ctl_parallel_read ) read_chunk_par( num_frame );
   else read_chunk_seq( num_frame );
   
   if(num_frame+frame_buffer_size < frames_read )
   {
	   mjpeg_error("Internal:readframe: %d internal error - buffer flushed too soon", frame_buffer_size );
	   abort();
   }

}

int readframe( int num_frame,
               unsigned char *frame[] )
{
   int n;

   load_frame( num_frame ); 
   n = num_frame % frame_buffer_size;
   frame[0] = frame_buffers[n][0];
   frame[1] = frame_buffers[n][1];
   frame[2] = frame_buffers[n][2];

   return 0;
}

#ifndef LINUX
int HasAltiVec()
{
    int mib[2], gHasAltivec;
    size_t len;

    mib[0] = CTL_HW;
    mib[1] = HW_VECTORUNIT;
    len = sizeof(gHasAltivec);
    sysctl(mib, 2, &gHasAltivec, &len, NULL, 0);
    return (gHasAltivec != 0);
}
#else
int HasAltiVec()
{
	return 0;
}
#endif

void init_read(void)
{
	uint32_t accel;
#ifndef LINUX
	if (HasAltiVec()) accel = MPEG2_ACCEL_PPC_ALTIVEC;
	else accel = 0; 
#else
	accel = 0;
#endif
	mpeg2_accel(accel);
	mpeg2dec = mpeg2_init();
	pictSize = 0;
	gotFirstPict = 0;
#ifdef DEMO
	demo_pic_count = 0;
#endif
}

#include <assert.h>
void read_stream_params(	int *opt_horizontal_size,
							int *opt_vertical_size, /* frame size (pels) */
							int *opt_aspectratio, /* aspect ratio information (pel or display) */
							int *opt_frame_rate_code, /* coded value of playback display frame rate */
							int *opt_video_format, /* component, PAL, NTSC, SECAM or MAC */
							int *opt_color_primaries, /* source primary chromaticity coordinates */
							int *opt_transfer_characteristics, /* opto-electronic transfer char. (gamma) */
							int *opt_matrix_coefficients, /* Eg,Eb,Er / Y,Cb,Cr matrix coefficients */
							int *opt_display_horizontal_size, 
							int *opt_display_vertical_size, /* display size */
							int *opt_profile,
							int *opt_level /* syntax / parameter constraints */
							)
{
	const mpeg2_sequence_t *s;
	int i;
	
	i = read(istrm_fd,rawbuf,RAWBUFSIZE);
	if (i <= 0) // original m2v stream is void
		exit(0); // so we just exit !
	ibitcount += i << 3; pictSize += i;
	mpeg2_buffer(mpeg2dec,rawbuf,rawbuf+i);
	mpeg2_parse(mpeg2dec);
	
	s = mpeg2_info(mpeg2dec)->sequence; assert(s);
	
	/*
	fprintf(stderr, "width: %d height: %d\n"
					"display_width: %d display_height: %d\n"
					"pixel_width: %d pixel_height: %d\n"
					"byte_rate: %d\n"
					"vbv_buffer_size: %d\n"
					//"aspect_code: %d\n"
					"frame_period: %d\n"
					"profile: %d level: %d\n"
					//"got color info: %d\n"
					//"colour_primaries: %d transfer_characteristics: %d matrix_coefficients: %d\n"
					"format: %d\n"
					//"isPAL: %d isNTSC: %d\n"
					,
					s->width, s->height,
					s->display_width, s->display_height,
					s->pixel_width, s->pixel_height,
					s->byte_rate,
					s->vbv_buffer_size,
					//s->aspect_code,
					s->frame_period,
					s->profile_level_id >> 4, s->profile_level_id & 0x0F,
					//(s->flags & SEQ_FLAG_COLOUR_DESCRIPTION),
					//s->colour_primaries, s->transfer_characteristics, s->matrix_coefficients,
					(s->flags & SEQ_MASK_VIDEO_FORMAT)
					//(s->frame_code == 3), (s->frame_code != 3)
					);
	*/
	
	*opt_horizontal_size = s->width;
	*opt_vertical_size = s->height;
	
	{
		int mw = s->display_width * s->pixel_width;
		int mh = s->display_height * s->pixel_height;
		// simplify
		
		int gcd = 0;
		while(gcd != 1)
		{
			int tw = mw;
			int th = mh;
			while (tw)
			{
				int tmp = tw;
				tw = th % tmp;
				th = tmp;
			}
			gcd = th;
			mw /= gcd;
			mh /= gcd;
		}
		
		if (mw == 4 && mh == 3) *opt_aspectratio = 2;
		else if (mw == 16 && mh == 9) *opt_aspectratio = 3;
		else *opt_aspectratio = 4; // 2.21:1
	}

	if (s->frame_period == 900900) *opt_frame_rate_code = 4; // 29.97 fps
	else if (s->frame_period == 1126125) *opt_frame_rate_code = 1; // 23.376 fps
	else *opt_frame_rate_code = 3; // 25 fps
	
	*opt_video_format = ((s->flags & SEQ_MASK_VIDEO_FORMAT) == SEQ_VIDEO_FORMAT_PAL) ? 1/*pal*/ : 2/*ntsc*/;
	
	if (s->flags & SEQ_FLAG_COLOUR_DESCRIPTION)
	{
		*opt_color_primaries = s->colour_primaries;
		*opt_transfer_characteristics = s->transfer_characteristics;
		*opt_matrix_coefficients = s->matrix_coefficients;
	}
	else
	{
		if (*opt_video_format == 1) // pal
			*opt_color_primaries = *opt_transfer_characteristics = *opt_matrix_coefficients = 5;
		else // ntsc
			*opt_color_primaries = *opt_transfer_characteristics = *opt_matrix_coefficients = 6;
	}
	*opt_display_horizontal_size = s->display_width; 
	*opt_display_vertical_size = s->display_height;
	*opt_profile = s->profile_level_id >> 4;
	*opt_level = s->profile_level_id & 0x0F;

}

void frame_pinfo(	int num_frame, int *type, int *tempref,
					int *prog, int *tff, int *repeat, /*double *quantiser,*/
					int64_t *bitcount, int *pictSize, char *seqHead)
{
	int n = num_frame;

	if(  last_frame > 0 && num_frame > last_frame ) n = last_frame;
	load_frame( n );

    if( last_frame > 0 && n > last_frame )  n = last_frame;
	
	n = n % frame_buffer_size;
	
	*type = pinfos[n].type;
	*tempref = pinfos[n].tempref;
	*prog = pinfos[n].prog;
	*tff = pinfos[n].tff;
	*repeat = pinfos[n].repeat;
	//*quantiser = pinfos[n].quantiser;
	*bitcount = pinfos[n].bitcount;
	*pictSize = pinfos[n].pictSize;
	*seqHead = pinfos[n].seqHead;
}
