
#include <config.h>
#include <stdio.h>
#include <stdlib.h>
#include <assert.h>
#include "global.h"
#include "vlc.h"

/* private prototypes */
static void putDC (const sVLCtable *tab, int val);

/* generate variable length code for luminance DC coefficient */
void putDClum(val)
int val;
{
  putDC(DClumtab,val);
}

/* generate variable length code for chrominance DC coefficient */
void putDCchrom(val)
int val;
{
  putDC(DCchromtab,val);
}

/* generate variable length code for DC coefficient (7.2.1) */
static void putDC(const sVLCtable *tab, int val)
{
  int absval, size;

  absval = abs(val);
  if (absval>opt_dctsatlim)
  {
    /* should never happen */
    mjpeg_error("Internal: DC value out of range (%d)",val);
	abort();
  }

  /* compute dct_dc_size */
  size = 0;

  while (absval)
  {
    absval >>= 1;
    size++;
  }

  /* generate VLC for dct_dc_size (Table B-12 or B-13) */
  putbits(tab[size].code,tab[size].len);

  /* append fixed length code (dc_dct_differential) */
  if (size!=0)
  {
    if (val>=0)
      absval = val;
    else
      absval = val + (1<<size) - 1; /* val + (2 ^ size) - 1 */
    putbits(absval,size);
  }
}

/* generate variable length code for first coefficient
 * of a non-intra block (7.2.2.2) */
void putACfirst(run,val)
int run,val;
{
  if (run==0 && (val==1 || val==-1)) /* these are treated differently */
    putbits(2|(val<0),2); /* generate '1s' (s=sign), (Table B-14, line 2) */
  else
    putAC(run,val,0); /* no difference for all others */
}

/* generate variable length code for other DCT coefficients (7.2.2) */
void putAC(run,signed_level,vlcformat)
int run,signed_level,vlcformat;
{
  int level, len;
  const VLCtable *ptab = NULL;

  level = abs(signed_level);

  /* make sure run and level are valid */
  if (run<0 || run>63 || level==0 || level>opt_dctsatlim)
  {
	  if( signed_level != -(opt_dctsatlim+1)) 	/* Negative range is actually 1 more */
	  {
		  mjpeg_error("Internal: AC value out of range (run=%d, signed_level=%d)",
					  run,signed_level);
		  abort();
	  }
  }

  len = 0;

  if (run<2 && level<41)
  {
    /* vlcformat selects either of Table B-14 / B-15 */
    if (vlcformat) ptab = &dct_code_tab1a[run][level-1];
    else ptab = &dct_code_tab1[run][level-1];

    len = ptab->len;
  }
  else if (run<32 && level<6)
  {
    /* vlcformat selects either of Table B-14 / B-15 */
    if (vlcformat) ptab = &dct_code_tab2a[run-2][level-1];
    else ptab = &dct_code_tab2[run-2][level-1];

    len = ptab->len;
  }

  if (len!=0) /* a VLC code exists */
  {
    putbits(ptab->code,len);
    putbits(signed_level<0,1); /* sign */
  }
  else
  {
    /* no VLC for this (run, level) combination: use escape coding (7.2.2.3) */
    putbits(1l,6); /* Escape */
    putbits(run,6); /* 6 bit code for run */
	putbits(signed_level,12);
  }
}

/* generate variable length code for macroblock_address_increment (6.3.16) */
void putaddrinc(addrinc)
int addrinc;
{
  while (addrinc>33)
  {
    putbits(0x08,11); /* macroblock_escape */
    addrinc-= 33;
  }
  assert( addrinc >= 1 && addrinc <= 33 );
  putbits(addrinctab[addrinc-1].code,addrinctab[addrinc-1].len);
}

/* generate variable length code for macroblock_type (6.3.16.1) */
void putmbtype(pict_type,mb_type)
int pict_type,mb_type;
{
  putbits(mbtypetab[pict_type-1][mb_type].code,
          mbtypetab[pict_type-1][mb_type].len);
}

/* generate variable length code for motion_code (6.3.16.3) */
void putmotioncode(motion_code)
int motion_code;
{
  int abscode;

  abscode = (motion_code>=0) ? motion_code : -motion_code; /* abs(motion_code) */
  putbits(motionvectab[abscode].code,motionvectab[abscode].len);
  if (motion_code!=0)
    putbits(motion_code<0,1); /* sign, 0=positive, 1=negative */
}

/* generate variable length code for dmvector[t] (6.3.16.3), Table B-11 */
void putdmv(dmv)
int dmv;
{
  if (dmv==0) putbits(0,1);
  else if (dmv>0) putbits(2,2);
  else putbits(3,2);
}

/* generate variable length code for coded_block_pattern (6.3.16.4)
 *
 * 4:2:2, 4:4:4 not implemented
 */
void putcbp(cbp)
int cbp;
{
  putbits(cbptable[cbp].code,cbptable[cbp].len);
}
