/*=========================================================================

  Program:   Insight Segmentation & Registration Toolkit
  Module:    $RCSfile: itkWeightedCovarianceCalculator.h,v $
  Language:  C++
  Date:      $Date: 2008-06-30 15:34:58 $
  Version:   $Revision: 1.9 $

  Copyright (c) Insight Software Consortium. All rights reserved.
  See ITKCopyright.txt or http://www.itk.org/HTML/Copyright.htm for details.

     This software is distributed WITHOUT ANY WARRANTY; without even 
     the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
     PURPOSE.  See the above copyright notices for more information.

=========================================================================*/
#ifndef __itkWeightedCovarianceCalculator_h
#define __itkWeightedCovarianceCalculator_h

#include "itkArray.h"
#include "itkVariableSizeMatrix.h"
#include "itkSampleAlgorithmBase.h"

namespace itk{ 
namespace Statistics{
  
/** \class WeightedCovarianceCalculator
 * \brief Calculates the covariance matrix of the target sample data
 * where each measurement vector has an associated weight value
 *
 * Recent API changes:
 * The static const macro to get the length of a measurement vector,
 * 'MeasurementVectorSize'  has been removed to allow the length of a measurement
 * vector to be specified at run time. It is now obtained from the input sample.
 * Please use the function GetMeasurementVectorSize() to obtain the length. 
 * The mean output is an Array rather than a Vector. The covariance matrix is 
 * represented by a VariableSizeMatrix rather than a Matrix.
  * \sa CovarianceCalculator SampleAlgorithmBase
 */

template< class TSample >
class WeightedCovarianceCalculator :
      public SampleAlgorithmBase< TSample >
{
public:
  /** Standard class typedefs. */
  typedef WeightedCovarianceCalculator Self;
  typedef SampleAlgorithmBase< TSample > Superclass;
  typedef SmartPointer<Self> Pointer;
  typedef SmartPointer<const Self> ConstPointer;

  /** Standard Macros */
  itkTypeMacro(WeightedCovarianceCalculator, SampleAlgorithmBase);
  itkNewMacro(Self);

  /** Length of a measurement vector */
  typedef typename Superclass::MeasurementVectorSizeType MeasurementVectorSizeType;
  
  /** Measurement vector typedef */
  typedef typename TSample::MeasurementVectorType MeasurementVectorType;
  
  /** Weight calculation function typedef */
  typedef FunctionBase< MeasurementVectorType, double > WeightFunctionType;

  /** Typedef for the mean output */
  typedef Array< double >                               MeanType;

  /** Typedef for Covariance output */
  typedef VariableSizeMatrix< double >                  OutputType;
  
  /** Array typedef for weights */
  typedef Array< double > WeightArrayType;

  /** Sets the weights using an array */
  void SetWeights(WeightArrayType* array);

  /** Gets the weights array */
  WeightArrayType* GetWeights( void );

  /** Sets the weights using an function the function should have a method, 
   * Evaluate(MeasurementVectorType&).  */
  void SetWeightFunction(WeightFunctionType* func);

  /** Gets the weight function */
  WeightFunctionType* GetWeightFunction( void );
  
  /** Sets the mean (input) */
  void SetMean(MeanType* mean);

  /** Gets the mean */
  MeanType* GetMean( void );

  /** Returns the covariance matrix of the target sample data */ 
  OutputType* GetOutput( void );

protected:
  WeightedCovarianceCalculator();
  virtual ~WeightedCovarianceCalculator();
  void PrintSelf(std::ostream& os, Indent indent) const;

  /** Calculates the covariance and save it */
  void GenerateData( void );

  void ComputeCovarianceWithGivenMean( void );
  
  void ComputeCovarianceWithoutGivenMean( void );

private:
  OutputType* m_Output;
  MeanType* m_Mean;
  MeanType* m_InternalMean;
  WeightArrayType* m_Weights;
  WeightFunctionType* m_WeightFunction;
}; // end of class
    
  } // end of namespace Statistics 
} // end of namespace itk 

#ifndef ITK_MANUAL_INSTANTIATION
#include "itkWeightedCovarianceCalculator.txx"
#endif

#endif

