/* GrandPerspective, Version 2.6.1 
 *   A utility for Mac OS X that graphically shows disk usage. 
 * Copyright (C) 2005-2022, Erwin Bonsma 
 * 
 * This program is free software; you can redistribute it and/or modify it 
 * under the terms of the GNU General Public License as published by the Free 
 * Software Foundation; either version 2 of the License, or (at your option) 
 * any later version. 
 * 
 * This program is distributed in the hope that it will be useful, but WITHOUT 
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or 
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for 
 * more details. 
 * 
 * You should have received a copy of the GNU General Public License along 
 * with this program; if not, write to the Free Software Foundation, Inc., 
 * 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA. 
 */

#import <Cocoa/Cocoa.h>

#define ITEM_SIZE  unsigned long long
#define FILE_COUNT  unsigned long long


@interface Item : NSObject {
}


/* Returns a memory zone that is intended for storing an Item tree. Depending on the preference
 * settings, it can be 1) the default zone, 2) a dedicated zone for all trees, 3) a dedicated
 * private zone for the given tree.
 *
 * Using a dedicated zone can be beneficial because the life-span of all objects in a tree is
 * identical, and typically long-lived. As short-lived, temporary objects needed during tree
 * construction are created in the default zone, they won't be mixed in memory. This should result
 * in far less (hardly any) memory fragmentation, which means that memory is used more efficiently.
 */
+ (NSZone *)zoneForTree;

/* Indicates if the zone should be disposed of when the tree is deallocated. This is the case when
 * the tree used a dedicated, private zone.
 */
+ (BOOL) disposeZoneAfterUse:(NSZone *)zone;

- (instancetype) initWithItemSize:(ITEM_SIZE)size NS_DESIGNATED_INITIALIZER;

/* Item size should not be changed once it is set. It is not "readonly" to enable DirectoryItem
 * subclass to set it later (once it knows its size).
 */
@property (nonatomic) ITEM_SIZE itemSize;
@property (nonatomic, readonly) FILE_COUNT numFiles;

// An item is virtual if it is not a file item (i.e. a file or directory).
@property (nonatomic, getter=isVirtual, readonly) BOOL virtual;

@end
