RCS_ID("$Id: FFImageLoader.m 562 2006-08-20 14:17:01Z ravemax $")

#import "FFImageLoader.h"
#import <QuickTime/ImageCompression.h>
#import <QuickTime/QuickTimeComponents.h>

// Exception source
static NSString* EXCEPTION_SOURCE = @"FFImageLoader";

// Overlay file extension
static NSString* ImageOverlaySuffix	= @"overlay";

// Determines the correct image importer
static inline BOOL _graphicImporter(NSString* filename, GraphicsImportComponent* gi) {
	CFURLRef	url;
	FSRef		fsRef;
	BOOL		ok;
	FSSpec		fsspecImage;
	OSStatus	err;
	
	// Path -> URL
	url = CFURLCreateWithFileSystemPath(NULL, (CFStringRef)filename, kCFURLPOSIXPathStyle, FALSE);
	if (!url)
		return FALSE;
	
	// CFURL -> FS Ref
	ok = CFURLGetFSRef(url, &fsRef);
	CFRelease(url);
	if (!ok)
		return FALSE;
	
	// FSRef -> FSSpec -> importer
	err = FSGetCatalogInfo(&fsRef, kFSCatInfoNone, NULL, NULL, &fsspecImage, NULL);
	if (err == noErr) 
		err = GetGraphicsImporterForFile(&fsspecImage, gi);
	
	return (err == noErr);
}

// Loads a image / thumb using Quicktime
// origWidth, origHeight may be NULL if searchThumb is false
static void _loadImage(NSString* filename, BOOL searchThumb,
					   GLint maxWidth, GLint maxHeight,
					   GLubyte** data, GLuint* origWidth, GLuint* origHeight,
					   GLuint* width, GLuint* height, GLuint* depth) {
	
	GraphicsImportComponent giComp; // component for importing image
    ImageDescriptionHandle  hImageDesc;
	BOOL					thumbFound;
	GWorldPtr				pGWorld = NULL;
	long					rowBytes;
	Rect					rectImage;
	GDHandle				origDevice;
	CGrafPtr				origPort;
	OSStatus				err;
	double					scale;
	
	// Filename -> importer
	if ((filename == nil) || !_graphicImporter(filename, &giComp))
		[NSException raise:EXCEPTION_SOURCE format:@"No graphics-importer or file doesn't exist -'%@'", filename];
	
	// Alloc mem for the image description
    hImageDesc = (ImageDescriptionHandle)NewHandle(sizeof(ImageDescriptionHandle));
    HLock((Handle)hImageDesc); //?
	
	// Search thumbnail
	if (searchThumb && (GraphicsImportSetImageIndexToThumbnail(giComp) != noThumbnailFoundErr)) {
		thumbFound = TRUE;
		
		// Size of the thumbnail
		if (GraphicsImportGetImageDescription (giComp, &hImageDesc) != noErr)
			[NSException raise:EXCEPTION_SOURCE format:@"Failed to retrieve the thumb description for '%@'", filename];
		
		*origWidth	= (GLuint)(*hImageDesc)->width;
		*origHeight	= (GLuint)(*hImageDesc)->height;
		
	} else
		thumbFound	= FALSE;
	
	// Now the size of the real image
	if (GraphicsImportGetImageDescription (giComp, &hImageDesc) != noErr)
		[NSException raise:EXCEPTION_SOURCE format:@"Failed to retrieve the image description for '%@'", filename];
	
	*width   = (GLuint)(*hImageDesc)->width;
	*height  = (GLuint)(*hImageDesc)->height;
	*depth   = (GLuint)(*hImageDesc)->depth;
	
	DisposeHandle((Handle)hImageDesc);
	
	// Is also the original size unless a thumb was found
	if (!thumbFound && (origWidth != NULL)) {
		*origWidth	= *width;
		*origHeight	= *height;
	}
	
	// Scaling
	if (maxWidth > 0) {
		scale = MIN((double)maxWidth / (double)*width, (double)maxHeight / (double)*height);
		if (scale < 1.0) {
			*width	*= scale;
			*height	*= scale;
		}
	}
	
	// Alloc mem
	rowBytes	= *width << 2; // 32bit
	*data		= malloc(*height * rowBytes);
	
	// Create GWorld (also assign "imagePtr" to GWorld)
	SetRect(&rectImage, 0, 0, (short)*width, (short)*height);
	QTNewGWorldFromPtr (&pGWorld, k32ARGBPixelFormat, &rectImage, NULL, NULL, 0, *data, rowBytes); 	
	if (NULL == pGWorld) {
		CloseComponent(giComp);
		[NSException raise:EXCEPTION_SOURCE format:@"QTNewGWorldFromPtr (internal) failed for '%@'", filename];
    }
	
	GetGWorld (&origPort, &origDevice); // save onscreen graphics port
										// decompress (draw) to gworld and thus fill buffer
	
	// Scale
	if ((maxWidth > 0) && (scale < 1.0)) {
		MatrixRecord	matrix;
		
		SetIdentityMatrix (&matrix);
		ScaleMatrix(&matrix,
					X2Fix(scale),	X2Fix(scale),
					X2Fix(0.0),		X2Fix(0.0));
		TranslateMatrix (&matrix, X2Fix(0.0), X2Fix(0.0));
		GraphicsImportSetMatrix(giComp, &matrix);
	}
	
	// Import
	err = GraphicsImportSetGWorld (giComp, pGWorld, NULL);
	if (err == noErr)
		err = GraphicsImportSetQuality (giComp, codecLosslessQuality);
	if ((err == noErr) && GetGWorldPixMap(pGWorld) && LockPixels(GetGWorldPixMap(pGWorld)))
		GraphicsImportDraw (giComp);
	else {
    	DisposeGWorld (pGWorld);
		CloseComponent(giComp);
		[NSException raise:EXCEPTION_SOURCE format:@"Import failed for '%@'", filename];
    }
	UnlockPixels (GetGWorldPixMap (pGWorld));
	CloseComponent(giComp);
	SetGWorld(origPort, origDevice); // set current graphics port to offscreen = restore 
									 // done with gworld and image since they are loaded to a texture
	DisposeGWorld (pGWorld);
}

void loadImage(NSString* filename, GLubyte** data, GLuint* width,
			   GLuint* height, GLuint* depth) {

	_loadImage(filename, FALSE, 0, 0, data, NULL, NULL, width, height, depth);
	
	// If not an exception has been raised
	NSString* olfname = [[NSString alloc] initWithFormat:@"%@.%@", filename, ImageOverlaySuffix];
	if ([[NSFileManager defaultManager] fileExistsAtPath:olfname]) {
		GLubyte*	olData; // ARGB
		GLuint		olWd, olHt, olDt;
		
		NS_DURING
			// Load the overlay. The fileformat is determined by Quicktime
			_loadImage(olfname, FALSE, 0, 0, &olData, NULL, NULL, &olWd, &olHt, &olDt);
			
			// Alpha-blending
			if ((olWd <= *width) && (olHt <= *height)) {
				GLubyte*	olPtr, *nPtr;
				unsigned	olNDiff;
				unsigned	i, j;
				GLubyte		al, alD;
				
				olPtr	= olData;
				nPtr	= *data;
				olNDiff	= (*width - olWd) * 4;
				
				for (j = 0; j < olHt; j++) {
					for (i = 0; i < olWd; i++) {
						al	= olPtr[0];
						alD	= 255 - al;
						
						nPtr[1] = ((alD * nPtr[1]) + al * olPtr[1]) / 255;
						nPtr[2] = ((alD * nPtr[2]) + al * olPtr[2]) / 255;			
						nPtr[3] = ((alD * nPtr[3]) + al * olPtr[3]) / 255;		
						
						olPtr += 4;
						nPtr += 4;
					}
					nPtr += olNDiff;
				}
			}
			
			free(olData);
		NS_HANDLER
			NSLog(@"Failed to load the overlay");
		NS_ENDHANDLER
	}
	
	[olfname release];
}

BOOL loadThumbnail(NSString* filename,
				   GLint thumbWidth, GLint thumbHeight,
				   NSImage** img,
				   GLuint* origWidth, GLuint* origHeight, GLuint* depth) {

	#pragma pack(1)
	typedef struct {
		unsigned char a, r, g, b;
	} FFARGB;
	#pragma pack()
	
	GLuint				wd, ht;
	GLubyte*			data;
	NSBitmapImageRep*	rep;
	FFARGB*				dataPtr, *endPtr;
	register FFARGB		argbTmp;
		
	// Try to load the thumb
	NS_DURING
		_loadImage(filename, TRUE, thumbWidth, thumbHeight, &data, origWidth, origHeight,
				   &wd, &ht, depth);
	NS_HANDLER
		return FALSE;
	NS_ENDHANDLER
	
	// Create a NSImageRep(resentation)
	rep = [[[NSBitmapImageRep alloc] 
			initWithBitmapDataPlanes:&data
						  pixelsWide:wd
						  pixelsHigh:ht
					   bitsPerSample:8
					 samplesPerPixel:4
							hasAlpha:TRUE
							isPlanar:FALSE
					  colorSpaceName:NSCalibratedRGBColorSpace
						 bytesPerRow:wd * 4
						bitsPerPixel:32] autorelease];		

	// RGBA
	dataPtr = (FFARGB*)[rep bitmapData];
	endPtr	= dataPtr + wd * ht;

	while (dataPtr < endPtr) {
		// ARGB -> RGBA (only work on PPC)
		// *dataPtr =  *((unsigned char*)dataPtr) + (*dataPtr << 8);

		argbTmp.r = dataPtr->g;
		argbTmp.g = dataPtr->b;
		argbTmp.b = dataPtr->a;
		argbTmp.a = dataPtr->r;
		
		*dataPtr = argbTmp;
		
		dataPtr++;
	}
	
	// Create the image
	*img = [[NSImage alloc] initWithData:[rep TIFFRepresentation]];
	
	// Now it should be safe to release the mem
	free(data);

	return TRUE;
}

